/** $Revision:   1.0  $ **/
/**/
/************************************************************************
*
*   Title: Sample Code for 1756-MVI CIP API
*
*   Abstract:
*
*   This sample code illustrates a variety of API functions,
*   while also performing a useful function.  This program implements
*   an interactive diagnostic utility that can be used to verify
*   communications between the MVI module and the controller.
*   The CIP API and Serial Port API are used.
*
*   Environment:    1756-MVI
*                   CIP API
*                   General Software DOS 6-XL
*                   Borland/Microsoft C/C++ Compiler (16-bit)
*
*                   Copyright (c) 1999-2000 Online Development, Inc.
*
*
************************************************************************/

/*=======================================================================
=                           INCLUDE FILES                               =
=======================================================================*/

#include    <stdio.h>
#include    <conio.h>
#include    <stdlib.h>
#include    <math.h>
#include    <dos.h>
#include    <ctype.h>
#include    <string.h>
#include    <time.h>
#include    "cipapi.h"
#include    "mvispapi.h"

/*=======================================================================
=                      LOCAL SUPPORT PROTOTYPES                         =
=======================================================================*/

void BPErrorExit(int);
void SPErrorExit(void);
void ComPutStr(char *);
void ComGetStr(char *);
void ParseCmdLine(int, char **);
int ProcessCommand(void);
int ShowHelp(char *);
int ShowCfg(char *);
int SetLed(char *);
int DumpOutImage(char *);
int WriteInImage(char *);
int DumpOutMsg(char *);
int WriteInMsg(char *);
int LoopBackIO(char *);

/* API Callbacks */
MVICALLBACK ConnectProc(MVIHANDLE objhandle, MVICIPCONNSTRUC *psConn);
MVICALLBACK ServiceProc(MVIHANDLE objhandle, MVICIPSERVSTRUC *psServ);

/*=======================================================================
=                    MODULE WIDE GLOBAL VARIABLES                       =
=======================================================================*/

/*** Backplane ***/
MVIHANDLE  handle;              /* Handle returned from MVIcip_Open function */
MVIHANDLE  ObjHandle;           /* Handle to Assembly Object */
MVIHANDLE  ConnHandle;          /* Handle received once connection established */
MVIHANDLE  StsConnHandle;       /* Handle of connection status */

BYTE *pDataBuffers = NULL;      /* pointer to buffer to hold all data */
BYTE *pRxDatConn = NULL;        /* Pointers to Scheduled Connection Data buffers */
BYTE *pTxDatConn = NULL;
BYTE *pCfgData = NULL;
BYTE *pMsgOutData = NULL;       /* Pointer to Unscheduled Message Output Data buffer */
BYTE *pMsgInData = NULL;        /* Pointer to Unscheduled Message Input Data buffer */
BYTE *pStsData = NULL;

WORD RxDatConnSize;             /* Data sizes for App Data */
WORD TxDatConnSize;
WORD CfgDataSize;
WORD MsgOutDataSize;
WORD MsgInDataSize;
WORD StsDataSize;

DWORD HostDeviceSn;             /* Serial number of host that makes a connection */
WORD  HostVendorId;             /* Vendor ID of host that makes a connection */

int fConnected = 0;             /* Connection status: 1=connected, 0=not */

/*** Serial Port ***/
int Commport;

/***  Commands  ***/
typedef struct
{
    char    *Cmd;
    int     (*CmdFunct)(char *);
} COMMANDSTRUCT;

COMMANDSTRUCT Commands[] =
{
    { "cfg", ShowCfg },
    { "do", DumpOutImage },
    { "wi", WriteInImage },
    { "dom", DumpOutMsg },
    { "wim", WriteInMsg },
    { "loop", LoopBackIO },
    { "led", SetLed },
    { "help", ShowHelp },
    { "?", ShowHelp },
    { "quit", NULL },
    { "q", NULL },
    { "", NULL }
};

#define MAXCMDSTRLEN    128     /* maximum length of user-input command string */

/*---------------------------------------------------------------------------
** App-defined instance numbers.
**  These instance values may be any non-zero value that the app desires.
**  However, the Generic Profile in the controller should match these values.
**  MSG instances should match the instance # of the MSG ladder instruction.
**---------------------------------------------------------------------------
*/
#define INPUT_INSTANCE  1       /* App-defined Instance # of Module Input data */
#define OUTPUT_INSTANCE 2       /* App-defined Instance # of Module Output data */
#define CONFIG_INSTANCE 4       /* App-defined Instance # of Module Config data */
#define STATUS_IN_INSTANCE 5    /* App-defined Instance # of Module Input Status */
#define STATUS_OUT_INSTANCE 6   /* App-defined Instance # of Module Output Status */
#define MSGIN_INSTANCE  7       /* App-defined Instance # of Module Message Input data */
#define MSGOUT_INSTANCE 8       /* App-defined Instance # of Module Message Output data */

/*---------------------------------------------------------------------------
** Size of this app's data tables
**
**  The data sizes may be any value that the app requires, up to 500bytes
**  for the Input instance and 496 bytes for the Output instance.
**  When the controller attempts to open a connection, it passes the
**  data size from the Generic Profile in the controller. An app can
**  decide whether to accept the controller's sizes or reject them if they
**  do not match. This app rejects connections whose sizes are greater than
**  the values defined here. (See the connect_proc Callback routine.)
**
**  Note: The 4 controller status bytes are not included in the controller's
**        Generic Profile.
**
**  For unscheduled Messages, the Input and Output Instance sizes may be up
**  to 500bytes each. However, the amount of data sent MUST match the Instance
**  size for every transaction, i.e. partial-data messages are not
**  supported yet. In this app, we define them to 200 to keep things simple.
**---------------------------------------------------------------------------
*/
#define RXSTSSIZE       4       /* 4bytes of controller status prepended to connected read data */
#define RXDATASIZE      496     /* App-defined Connected Read Data Size (from host) */
#define TXDATASIZE      500     /* App-defined Connected Write Data Size (to host) */
#define CFGDATASIZE     64      /* App-defined Configuration Data Size */
#define MSGOUTDATASIZE  200     /* App-defined Unscheduled Message data size */
#define MSGINDATASIZE   200     /* App-defined Unscheduled Message data size */
#define STATUSDATASIZE  20      /* Status Data Size */
#define DATABUFFERSIZE  (RXSTSSIZE + RXDATASIZE + TXDATASIZE + CFGDATASIZE + \
                         MSGOUTDATASIZE + MSGINDATASIZE + STATUSDATASIZE)

/*=======================================================================
=                       LOCAL SUPPORT ROUTINES                          =
=======================================================================*/

/***********************************************************************/
void ComPutStr(char *str)
{
    if (MVIsp_Puts(Commport,(BYTE *)str,0,NULL,1000L) != MVI_SUCCESS)
        SPErrorExit();
}

/***********************************************************************/
void ComGetStr(char *str)
{
    BYTE ch;
    int count;

    count = 0;
    do
    {
        if (MVIsp_Getch(Commport,&ch,TIMEOUT_FOREVER) != MVI_SUCCESS)
            SPErrorExit();
        switch (ch)
        {
            case '\r':      /* cr */
                MVIsp_Putch(Commport,ch,1000L);
                *str++ = '\0';
                break;

            case '\b':      /* backspace */
                if (count > 0)
                {
                    count--;
                    *--str = '\0';
                    MVIsp_Putch(Commport,ch,1000L);
                    MVIsp_Putch(Commport,' ',TIMEOUT_ASAP);
                    MVIsp_Putch(Commport,'\b',TIMEOUT_ASAP);
                }
                break;

            default:
                MVIsp_Putch(Commport,ch,1000L);
                if (count < (MAXCMDSTRLEN - 1))
                {
                    *str++ = ch;
                    count++;
                }
                break;
        }
    }
    while(ch != '\r' );
}

/***********************************************************************/
void BPErrorExit(int errcode)
{
    char errbuf[80];

    MVIcip_ErrorString(errcode, errbuf);    /* get text description of error */
    ComPutStr(errbuf);
    ComPutStr("\n\r");
    MVIcip_UnregisterAssemblyObj(handle, ObjHandle);    /* tidy up */
    MVIcip_Close(handle);                   /* should always close before exiting */
    MVIsp_Close(Commport);
    exit(1);
}

/***********************************************************************/
void SPErrorExit(void)
{
    int count;

    /* if serial port error, can't rely on error messages */
    MVIcip_UnregisterAssemblyObj(handle, ObjHandle);    /* tidy up */
    MVIcip_Close(handle);                   /* should always close before exiting */
    do
    {
        MVIsp_GetCountUnsent(Commport,&count);
    }
    while (count);
    MVIsp_Close(Commport);
    exit(1);
}


/*=======================================================================
=                       MAIN ENTRY POINT                                =
=======================================================================*/

/************************************************************************
*
*     Entry point:                                                      
*       main                                           
*
*     Description:                                                      
*       This is the entry point of the sample program.  A variety of
*       API functions are called to illustrate usage.
*
*     Arguments:                                                        
*       none
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       none
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void main(int argc, char *argv[])
{
    int     CmdSts;
    int     rc;
    char    tmpbuf[50];
    time_t  tm;

    Commport = COM1;
    ConnHandle = (MVIHANDLE)NULL;
    StsConnHandle = (MVIHANDLE)NULL;

    ParseCmdLine(argc,argv);

    /* Open a serial port console */
    rc = MVIsp_Open(Commport,BAUD_19200,PARITY_NONE,WORDLEN8,STOPBITS1);
    if (rc != MVI_SUCCESS)
    {
        printf("Error Opening Serial Port %d\n",rc);
        exit(1);
    }

    /* print a sign-on message */
    ComPutStr("\n\r1756-MVI Sample Program V0.3");
    ComPutStr("\n\r--Enter 'help' or '?' to show menu\n\r");

    /*
    ** Allocate and initialize the application's connected data buffers
    */

    pDataBuffers = (BYTE *)malloc(DATABUFFERSIZE);
    if( pDataBuffers == NULL )
    {
        ComPutStr("\n\rCan't Alloc Memory\n\r");
        SPErrorExit();
    }

    /* the 4bytes of controller status are included in the read connected data */
    pRxDatConn = &(pDataBuffers[0]);
    RxDatConnSize = RXDATASIZE + RXSTSSIZE;

    /* write connected data buffer */
    pTxDatConn = &(pDataBuffers[RxDatConnSize]);
    TxDatConnSize = TXDATASIZE;

    /* config data buffer */
    pCfgData = &(pDataBuffers[RxDatConnSize + TxDatConnSize]);
    CfgDataSize = CFGDATASIZE;

    /* Unscheduled msg output data buffer */
    pMsgOutData = &(pDataBuffers[RxDatConnSize + TxDatConnSize + CfgDataSize]);
    MsgOutDataSize = MSGOUTDATASIZE;

    /* Unscheduled msg input data buffer */
    pMsgInData = &(pDataBuffers[RxDatConnSize + TxDatConnSize + CfgDataSize + MsgOutDataSize]);
    MsgInDataSize = MSGINDATASIZE;

    /* Status data buffer */
    pStsData = &(pDataBuffers[RxDatConnSize + TxDatConnSize + CfgDataSize + MsgOutDataSize + MsgInDataSize]);
    StsDataSize = STATUSDATASIZE;

    /* init the buffers to 0 */
    memset( pDataBuffers, 0, DATABUFFERSIZE );

    /* Open the backplane API */
    if (MVI_SUCCESS != (rc = MVIcip_Open(&handle)))
    {
        sprintf(tmpbuf, "\n\rMVIcip_Open failed: %d", rc);
        ComPutStr(tmpbuf);
        sprintf(tmpbuf, "\n\rBackplane driver mvi56bp.exe is not loaded");
        ComPutStr(tmpbuf);
    }
    else
    {
        /* To allow a connection between the controller and the API, an 
        ** assembly object must be registered with the API. This links
        ** the app with the connection.
        */
        if (MVI_SUCCESS != (rc = MVIcip_RegisterAssemblyObj(handle, &ObjHandle,
                                    0L, ConnectProc, ServiceProc, NULL)))
        {
            sprintf(tmpbuf,"\n\rMVIcip_RegisterAssemblyObj failed: %d\r\n", rc);
            ComPutStr(tmpbuf);
        }

        /* wait for connection with controller */
        tm = time(NULL);
        while (1)
        {
            if (fConnected)
                break;
            if ((time(NULL) - tm) >= 6)
            {
                ComPutStr("\r\nWARNING: Timed out waiting for Host connection");
                ComPutStr("\r\n         Use the 'cfg' command to check\r\n");
                break;
            }
        }
    }

    /*
    ** this app is now online, process user's commands
    */
    do
    {
        /* display prompt */
        ComPutStr("\n\rmvi> ");

        /* get a command from user */
        CmdSts = ProcessCommand();

    } while (CmdSts == 0);

    /* Unregister the assembly object and Close the API's
    ** No need to print any error messages here since we
    ** are closing the API's
    */
    MVIcip_UnregisterAssemblyObj(handle, ObjHandle);
    MVIsp_Close(Commport);      /* should always close before exiting */
    MVIcip_Close(handle);
}

/******************************************************************************
******************************************************************************/
int ProcessCommand(void)
{
    static char cCmdBuf[MAXCMDSTRLEN];
    char    tmpbuf[100];
    char    *szArgs;
    char    *szCmd;
    COMMANDSTRUCT *pCommands;
    int CmdSts;

    memset(cCmdBuf,0,100);
    ComGetStr(cCmdBuf);

    /* process command */
    szCmd = strtok(cCmdBuf," \t\r");
    if (strlen(szCmd) == 0)
        return(0);   /* re-prompt if user keeps pressing cr */

    pCommands = (COMMANDSTRUCT *)&Commands;
    while (strlen(pCommands->Cmd) > 0)
    {
        if (stricmp(pCommands->Cmd,szCmd) == 0)
        {
            if (pCommands->CmdFunct != NULL)
            {
                szArgs = &(cCmdBuf[strlen(szCmd)+1]);
                CmdSts = (pCommands->CmdFunct)(szArgs);
                if (CmdSts == 1)
                    ComPutStr("\n\rError #1 -- Bad Syntax");
                else if (CmdSts > 1)
                {
                    sprintf(tmpbuf,"\n\rError %d",CmdSts);
                    ComPutStr(tmpbuf);
                }
            }
            else if (szCmd[0] == 'q' || szCmd[0] == 'Q')
                return(1);  /* exit the program */
            return(0);
        }
        pCommands++;
    }
    /* command not found */
    ComPutStr("\n\rHuh? ('?' gives Command List)");
    return(0);
}

/******************************************************************************
******************************************************************************/
//#pragma argsused
int ShowHelp(char *tmp)
{
    ComPutStr("\n\r\n\rCommands:");
    ComPutStr("\n\r----------------------");
    ComPutStr("\n\rdo                - dump connected output data");
    ComPutStr("\n\rwi byte# val      - write val(hex) to connected input byte");
    ComPutStr("\n\rdom               - dump message output data");
    ComPutStr("\n\rwim byte# val     - write val(hex) to message input byte");
    ComPutStr("\n\rled [1|2] [0|1]   - turn LED 1 or 2 off(0) or on(1)");
    ComPutStr("\n\rcfg               - show the io configuration");
    ComPutStr("\n\rloop              - loopback first 10 output bytes to input");
    ComPutStr("\n\rhelp,?            - print this Command List");
    ComPutStr("\n\rq,quit            - exit this program\n\r");
    tmp++;      /* suppress compiler warning */
    return(0);
}

/******************************************************************************
******************************************************************************/
int DumpOutImage(char *szTmp)
{
    int rc;
    int i;
    char s[64];

    szTmp++;        /* suppress compiler warning */

    if (MVI_SUCCESS != (rc = MVIcip_ReadConnected(handle, ConnHandle, pRxDatConn,
                                        RXSTSSIZE, RxDatConnSize - RXSTSSIZE)))
    {
        sprintf(s,"\n\rMVIcip_ReadConnected failed: %d",rc);
        ComPutStr(s);
        return(0);
    }

    for (i=0; i<(int)(RxDatConnSize-RXSTSSIZE); i++)
    {
        if ((i % 16) == 0)
        {
            sprintf(s,"\n\rWord %3d:",i);
            ComPutStr(s);
        }
        sprintf(s," %02X",pRxDatConn[i]);
        ComPutStr(s);
    }

    return(0);
}

/******************************************************************************
******************************************************************************/
int WriteInImage(char *szTmp)
{
    int rc;
    char    *s;
    int wordnum;
    BYTE val;
    char ss[64];

    /* parse command for word# and val */
    s = strtok(szTmp," \t\r");
    if (strlen(s) == 0)
        return(1);          /* syntax error */
    wordnum = atoi(s);
    s = strtok(NULL," \t\r");
    if (strlen(s) == 0)
        return(1);          /* syntax error */
    val = (BYTE)strtoul(s,NULL,16);

//    pTxDatConn[wordnum] = val;
    if (MVI_SUCCESS != (rc = MVIcip_WriteConnected(handle, ConnHandle, &val, wordnum, 1)))
    {
        sprintf(ss,"\n\rMVIcip_WriteConnected failed: %d",rc);
        ComPutStr(ss);
    }

    return(0);
}

/******************************************************************************
******************************************************************************/
int DumpOutMsg(char *szTmp)
{
    WORD i;
    char s[64];

    szTmp++;        /* suppress compiler warning */

    if (!fConnected)
    {
        ComPutStr("\n\rNot Connected\n\r");
        return(0);
    }

    for (i=0; i<(MsgOutDataSize); i++)
    {
        if ((i % 16) == 0)
        {
            sprintf(s,"\n\rWord %3d:",i);
            ComPutStr(s);
        }
        sprintf(s," %02X",pMsgOutData[i]);
        ComPutStr(s);
    }

    return(0);
}

/******************************************************************************
******************************************************************************/
int WriteInMsg(char *szTmp)
{
    char    *s;
    WORD wordnum;
    BYTE val;

    /* parse command for word# and val */
    s = strtok(szTmp," \t\r");
    if (strlen(s) == 0)
        return(1);          /* syntax error */
    wordnum = (WORD)strtoul(s,NULL,10);
    s = strtok(NULL," \t\r");
    if (strlen(s) == 0)
        return(1);          /* syntax error */
    val = (BYTE)strtoul(s,NULL,16);

    if (!fConnected)
    {
        ComPutStr("\n\rNot Connected\n\r");
        return(0);
    }

    if (wordnum >= MsgInDataSize)
        return(1);
    pMsgInData[wordnum] = val;

    return(0);
}

/******************************************************************************
******************************************************************************/
int LoopBackIO(char *szTmp)
{
    int rc;
    BYTE bch;
    char s[64];

    szTmp++;        /* suppress compiler warning */

    ComPutStr("\n\rPress a key to end");

    do
    {
        if (MVI_SUCCESS != (rc = MVIcip_ReadConnected(handle, ConnHandle, pRxDatConn,
                                        0, 10 + RXSTSSIZE)))
        {
            sprintf(s,"\n\rMVIcip_ReadConnected failed: %d\n\r", rc);
            ComPutStr(s);
            return(0);
        }

        /* copy 1st 10 bytes of output data to input buffer */
        memcpy(pTxDatConn,&(pRxDatConn[RXSTSSIZE]),10);

        if (MVI_SUCCESS != (rc = MVIcip_WriteConnected(handle, ConnHandle, pTxDatConn, 0, 10)))
        {
            sprintf(s,"\n\rMVIcip_WriteConnected failed: %d\n\r", rc);
            ComPutStr(s);
            return(0);
        }
        rc = MVIsp_Getch(Commport,&bch,TIMEOUT_ASAP);

        /* copy 1st 10 bytes of MSG output to MSG input */
        memcpy(pMsgInData,pMsgOutData,10);

    } while (rc == MVI_ERR_TIMEOUT);

    return(0);
}

/******************************************************************************
******************************************************************************/
//#pragma argsused
int SetLed(char *szTmp)
{
    int led,state;
    char *s;
    int rc;
    char ss[50];

    led = 0;
    state = 0;

    s = strtok(szTmp," \t\r");
    led = atoi(s);
    s = strtok(NULL," \t\r");
    state = atoi(s);

    if ( (led != 1) && (led != 2))
        return(1);
    if ( (state != 0) && (state != 1))
        return(1);

    if (MVI_SUCCESS != (rc = MVIcip_SetUserLED(handle, led, state)))
    {
        sprintf(ss,"\n\rSetUserLed Failed: %d",rc);
        ComPutStr(ss);
    }
    return(0);
}

/********************************************************************
********************************************************************/

int ShowCfg(char *szTmp)
{
    int rc;
    MVICIPVERSIONINFO verinfo;
    MVISPVERSIONINFO spver;
    MVICIPIDOBJ idobj;
    char tmpbuf[100];

    szTmp++;        /* suppress compiler warning */

    /*
    ** Display the Serial Port API version
    */
    if (MVI_SUCCESS != (rc = MVIsp_GetVersionInfo(&spver)))
    {
        sprintf(tmpbuf,"\n\rMVIsp_GetVersionInfo failed: %d\n\r", rc);
        ComPutStr(tmpbuf);
    }
    sprintf(tmpbuf,"\n\r\n\rSerial Port API Version %d.%d", spver.APISeries, spver.APIRevision);
    ComPutStr(tmpbuf);

    /*
    ** Display the Backplane driver version
    */
    if (MVI_SUCCESS != (rc = MVIcip_GetVersionInfo(handle, &verinfo)))
    {
        sprintf(tmpbuf,"\n\rMVIcip_GetVersionInfo failed: %d\n\r", rc);
        ComPutStr(tmpbuf);
        return(0);
    }
    sprintf(tmpbuf,"\n\rBackplane Driver Version %d.%d",verinfo.BPDDSeries, verinfo.BPDDRevision);
    ComPutStr(tmpbuf);

    /*
    ** Display the CIP Backplane API version
    */
    sprintf(tmpbuf,"\n\rCIP BP API Version %d.%d", verinfo.APISeries, verinfo.APIRevision);
    ComPutStr(tmpbuf);

    /* Display the ID Object (module) information */
    if (MVI_SUCCESS != (rc = MVIcip_GetIdObject(handle, &idobj)))
    {
        sprintf(tmpbuf,"\n\rMVIcip_GetIdObject failed: %d\n\r", rc);
        ComPutStr(tmpbuf);
        return(0);
    }
    ComPutStr("\n\r\n\rModule Name: ");
    ComPutStr((char *)idobj.Name);
    sprintf(tmpbuf,"\n\rVendorID: %2X    DeviceType: %d",idobj.VendorID,idobj.DeviceType);
    ComPutStr(tmpbuf);
    sprintf(tmpbuf,"\n\rProdCode: %d    SerialNum:  %ld",idobj.ProductCode,idobj.SerialNo);
    ComPutStr(tmpbuf);
    sprintf(tmpbuf,"\n\rRevision: %d.%d",idobj.MajorRevision,idobj.MinorRevision);
    ComPutStr(tmpbuf);

    /* Display Connection Status */
    ComPutStr("\n\r\n\r");
    if (!fConnected)
        ComPutStr("Not ");
    ComPutStr("Connected with Host");

    if (fConnected)
    {
        /* show run/program mode */
        MVIcip_ReadConnected(handle,ConnHandle,pRxDatConn,0,RXSTSSIZE);
        ComPutStr("\n\r");
        if (*((WORD *)pRxDatConn) & 1)
            ComPutStr("Host in Run Mode");
        else
            ComPutStr("Host in Program Mode");

        /* Display the IO configuration */
        sprintf(tmpbuf,"\n\rInstance Sizes (bytes)  Input: %d  Output: %d",
            TxDatConnSize, RxDatConnSize - RXSTSSIZE);
        ComPutStr(tmpbuf);
    }

    ComPutStr("\n\r");
    return(0);
}

/********************************************************************
********************************************************************/
void ParseCmdLine(int argc, char *argv[])
{
    int     n;
    char    *arg;
    char    ch;
    
    for (n=1; n<argc; n++)
    {
        arg = argv[n];

        if ((*arg == '/') || (*arg == '-'))
        {
            ch = *(++arg);
            ch = toupper(ch);
            arg++;
            switch (ch)
            {
                case 'C':   /* -c2 for PRT2 */
                    Commport = atoi(arg) - 1;   /* PRT1=0, PRT2=1 */
                    break;
            }
        }
    }
}

/*=======================================================================
=                       BACKPLANE CALLBACK ROUTINES
=======================================================================*/

/*
** Stack Checking must be disabled for callback routines !!
*/
#ifdef __BORLANDC__
#pragma option -N-
#endif
#ifdef _MSC_VER
#pragma check_stack (off)
#endif

/************************************************************************
*
*     Entry point:
*       ConnectProc
*
*     Description:
*       This is the scheduled connection callback routine. This routine
*       is called by the API when a scheduled connection is opened or
*       closed.
*
*     Arguments:
*       objhandle           MVIHANDLE
*               handle of the registered object instance to which a
*               connection is being opened or closed.
*
*       psConn              MVICIPCONNSTRUC *
*               pointer to a structure containing connection information
*
*     External effects:
*       MVIcip_ReadConnected and MVIcip_WriteConnected functionality is
*       not valid until a connection has been opened via this callback.
*
*     Return value:
*       MVI_SUCCESS if connection is closing or if we accept the open
*       MVI_CIP_xxx otherwise
*
*-----------------------------------------------------------------------
*     Notes:
*       When a controller attempts to establish a scheduled data
*       connection with the app's assembly object or when it closes
*       the connection, this routine is called to notify the app.
*
*       Once a connection is open, the app can read/write to the
*       assembly object using the ReadConnected/WriteConnected API
*       functions.
*
*       In this sample app, we accept the connection if the requested
*       data sizes are less than or equal to what we require. We also
*       set a status flag to tell our app that the connection is open
*       or closed.
*
*       The __loadds keyword is required here to properly set the
*       DS segment register. This is necessary since this callback
*       is called from the API, which does not know where this app's
*       data segment is located. (Microsoft/Borland compatible)
*
************************************************************************/
MVICALLBACK ConnectProc(MVIHANDLE objhandle, MVICIPCONNSTRUC *psConn)
{
    if( ObjHandle != objhandle )
    {
        /*
        ** Somehow the object handles don't match. This
        ** should never happen. If it does return error.
        */
        return( MVI_CIP_FAILURE );
    }

    switch( psConn->reason )
    {
        case MVI_CIP_CONN_OPEN:
            /*
            ** Check the Instance number passed in the forward open. This
            ** number may or may not be used by the application. If the
            ** application supports multiple connections then the Instance
            ** can be used to determine what to connect to. The instance
            ** also will specify what instance the configuration data
            ** passed in the open is intended for.
            */
            if( psConn->instance != CONFIG_INSTANCE )
            {
                return( MVI_CIP_BAD_INSTANCE );
            }

            /*
            ** Check the connection points to determine if this is
            ** a status or data connection.
            */
            if( (psConn->producerCP == INPUT_INSTANCE) &&
                (psConn->consumerCP == OUTPUT_INSTANCE) )
            {
                /*
                ** Only one connection to be active at a time
                */
                if( ConnHandle != (MVIHANDLE)NULL )
                {
                    *psConn->extendederr = MVI_CIP_EX_CONNECTION_USED;
                    return( MVI_CIP_FAILURE );
                }

                /*
                ** If a Status connection is already open, the Data connection
                ** must match the Vendor ID and Serial number of the originator.
                */
                if( StsConnHandle != (MVIHANDLE)NULL )
                {
                    if( (psConn->lODeviceSn != HostDeviceSn ) ||
                        (psConn->iOVendorId != HostVendorId) )
                    {
                        *psConn->extendederr = MVI_CIP_EX_CONNECTION_USED;
                        return( MVI_CIP_FAILURE );
                    }
                }

                /*
                ** check the size of the txdata and rxdata
                ** Note: Remember to account for the 4 status bytes
                **       containing the Run/Program bit in the rxdata.
                */
                if( (psConn->txDataSize > TXDATASIZE) ||
                    (psConn->rxDataSize > (RXDATASIZE + RXSTSSIZE)) )
                {
                    *psConn->extendederr = MVI_CIP_EX_BAD_SIZE;
                    return( MVI_CIP_FAILURE );
                }

                /*
                ** Copy configuration data here, if present
                */
                if( (psConn->configData != NULL) && (psConn->configSize != 0) )
                {
                    /* check for valid size */
                    if( psConn->configSize > CFGDATASIZE )
                    {
                       *psConn->extendederr = MVI_CIP_EX_BAD_SIZE;
                       return( MVI_CIP_FAILURE );
                    }
                    memcpy( pCfgData, psConn->configData, psConn->configSize );
                }

                /*
                ** The connection handle being passed is a unique identifier
                ** for this connection. This value is used by the Send and
                ** read connected functions to identify which connections data
                ** is being accessed.
                */
                ConnHandle = psConn->connHandle;
                HostDeviceSn = psConn->lODeviceSn;
                HostVendorId = psConn->iOVendorId;
                TxDatConnSize = psConn->txDataSize;
                RxDatConnSize = psConn->rxDataSize;
                CfgDataSize = psConn->configSize;
            }

            else if( (psConn->producerCP == STATUS_IN_INSTANCE) &&
                     (psConn->consumerCP == STATUS_OUT_INSTANCE) )
            {
                /*
                ** If a Data connection is already open, the Status connection
                ** must match the Vendor ID and Serial number of the originator.
                */
                if( ConnHandle != (MVIHANDLE)NULL )
                {
                    if( (psConn->lODeviceSn != HostDeviceSn ) ||
                        (psConn->iOVendorId != HostVendorId) )
                    {
                        *psConn->extendederr = MVI_CIP_EX_CONNECTION_USED;
                        return( MVI_CIP_FAILURE );
                    }
                }

                /*
                ** Only one connection to be active at a time
                */
                if( StsConnHandle != (MVIHANDLE)NULL )
                {
                    *psConn->extendederr = MVI_CIP_EX_CONNECTION_USED;
                    return( MVI_CIP_FAILURE );
                }

                /*
                ** check the size of the txdata and rxdata
                ** Note: Remember to account for the 4 status bytes
                **       containing the Run/Program bit in the rxdata.
                */
                if( (psConn->txDataSize > STATUSDATASIZE) ||
                    (psConn->rxDataSize != 0) )
                {
                    *psConn->extendederr = MVI_CIP_EX_BAD_SIZE;
                    return( MVI_CIP_FAILURE );
                }

                /*
                ** The connection handle being passed is a unique identifier
                ** for this connection. This value is used by the Send and
                ** read connected functions to identify which connections data
                ** is being accessed.
                */
                StsConnHandle = psConn->connHandle;
                HostDeviceSn = psConn->lODeviceSn;
                HostVendorId = psConn->iOVendorId;
                StsDataSize = psConn->txDataSize;
            }

            else
            {
                return( MVI_CIP_FAILURE );
            }
            break;

        case MVI_CIP_CONN_OPEN_COMPLETE:
            /*
            ** The connection is now up and running. Load initial
            ** transmit data now. Also now is the time to notify
            ** the application threads that the connection is running.
            **
            ** When the host's profile for this module is configured for Data
            ** with Status, two connections are received: Data and Status.
            ** A connection with Status info can be received before the
            ** Data connection. In this sample app, we only care about the
            ** Data connection so check for it.
            **
            ** When the host's profile for this module is configured for Data only,
            ** only 1 connection (to I/O Data) will be received.
            */
            if (ConnHandle != (MVIHANDLE)NULL)     /* make sure this is I/O complete, not Sts */
            {
                MVIcip_WriteConnected(handle, ConnHandle, pTxDatConn, 0, TXDATASIZE);
                fConnected = 1;     /* indicate connection is now open */
            }
            break;

        case MVI_CIP_CONN_CLOSE:
            /*
            ** The connection handle being passed is a unique identifier
            ** for this connection. If it doesn't match this is not our
            ** connection.
            */
            if ( psConn->connHandle == ConnHandle )
            {
                /* let's go ahead and close the connection */
                /*
                ** Here we need to:
                **    Free and allocated resources.
                **    Clear any handles
                **    Stop the scheduled data updates
                */
                ConnHandle = (MVIHANDLE)NULL;

                /*
                ** In this app, we only care about the Data connection. So
                ** indicate the connection is closed only when the Data
                ** connection closes.
                */
                fConnected = 0;
            }
            else if( psConn->connHandle == StsConnHandle )
            {
                /* let's go ahead and close the connection */
                /*
                ** Here we need to:
                **    Free and allocated resources.
                **    Clear any handles
                **    Stop the scheduled data updates
                */
                StsConnHandle = (MVIHANDLE)NULL;
            }

            break;

        default:
            return( MVI_CIP_FAILURE );

    } /* end switch( psConn->reason ) */

    return( MVI_SUCCESS );
}

/************************************************************************
*
*     Entry point:
*       ServiceProc
*
*     Description:
*       This is the unscheduled message callback routine. This routine
*       is called by the API when an unscheduled message for the
*       registered assembly object is received.
*
*     Arguments:
*       objhandle           MVIHANDLE
*               handle of the registered object instance to which a
*               connection is being opened or closed.
*
*       psServ              MVICIPSERVSTRUC *
*               pointer to a structure containing message information
*
*     External effects:
*       If we accept a message, we read/write the data to/from our
*       buffers in this callback. This routine is a callback, and it
*       is called from the API asynchronously to the main routines.
*
*     Return value:
*       MVI_SUCCESS if message accepted
*       MVI_CIP_xxx otherwise
*
*-----------------------------------------------------------------------
*     Notes:
*       A controller uses the MSG ladder instruction to send an
*       unscheduled message to the app's assembly object. When the
*       API receives the message, this callback routine is called and
*       a response is returned to the controller.
*
*       It is important to understand that scheduled connections and
*       unscheduled messages to same instances of the assembly object
*       should access the same data. To access the data using
*       unscheduled messages only, the main app should not call
*       Read/WriteConnected and instead update data buffers here.
*       To access a separate set of data, separate instances should
*       be used with unscheduled messages.
*
*       This callback is mandatory. If the app does not need to
*       support unscheduled messages (i.e. no MSG ladder instruction,)
*       this routine should just return with an error, e.g.
*       MVI_CIP_BAD_SERVICE.
*
*       Data must be exchanged from within this callback routine.
*
*       In this sample app, we simply update the requested data.
*
*       The __loadds keyword is required here to properly set the
*       DS segment register. This is necessary since this callback
*       is called from the API, which does not know where this app's
*       data segment is located. (Microsoft/Borland compatible)
*
************************************************************************/
MVICALLBACK ServiceProc(MVIHANDLE objhandle, MVICIPSERVSTRUC *psServ)
{

    if( ObjHandle != objhandle )
    {
        /*
        ** Somehow the object handles don't match. This
        ** should never happen. If it does return error.
        */
        return( MVI_CIP_FAILURE );
    }

    /*
    ** Process according to the instance attribute.
    */

    switch( psServ->attribute )
    {
        case MVI_CIP_IA_DATA:

            /*
            ** Check the Service code.
            */
            switch( psServ->serviceCode )
            {
                case MVI_CIP_SC_SET_ATTR_SINGLE:
                    /*
                    ** This Service Code sets the entire instance with new data.
                    ** The MVI_CIP_SC_SET_MEMBER is used for partial data.
                    */

                    if( *psServ->msgSize != 0 )
                    {
                        switch( psServ->instance )
                        {
                            case MSGIN_INSTANCE:
                                return( MVI_CIP_ATTR_NOT_SETTABLE );

                            case MSGOUT_INSTANCE:
                                /* Must be entire instance data, no partial sizes */
                                if( *psServ->msgSize != MSGOUTDATASIZE )
                                {
                                    return( MVI_CIP_PARTIAL_DATA );
                                }
                                memcpy(pMsgOutData,*(psServ->msgBuf),MSGOUTDATASIZE);
                                break;

                            case CONFIG_INSTANCE:
                                if( *psServ->msgSize != CfgDataSize )
                                {
                                    return( MVI_CIP_PARTIAL_DATA );
                                }
                                memcpy(pCfgData, *(psServ->msgBuf), CfgDataSize);
                                break;

                            case STATUS_IN_INSTANCE:
                                return( MVI_CIP_ATTR_NOT_SETTABLE );

                            default:
                                return( MVI_CIP_BAD_INSTANCE );

                        } /* end switch( instance ) */
                    }
                    break;

                case MVI_CIP_SC_GET_ATTR_SINGLE:
                    /*
                    ** Return the data for the instance specified.
                    ** Must be entire instance data, no partial data sizes.
                    */
                    switch( psServ->instance )
                    {
                        case MSGIN_INSTANCE:
                            *psServ->msgBuf   = pMsgInData;
                            *psServ->msgSize  = MSGINDATASIZE;
                            break;

                        case MSGOUT_INSTANCE:
                            *psServ->msgBuf   = pMsgOutData;
                            *psServ->msgSize  = MSGOUTDATASIZE;
                            *psServ->msgSize  = 0;
                            break;

                        case CONFIG_INSTANCE:
                            *psServ->msgBuf   = pCfgData;
                            *psServ->msgSize  = CfgDataSize;
                            break;

                        case STATUS_IN_INSTANCE:
                            *psServ->msgBuf   = pStsData;
                            *psServ->msgSize  = StsDataSize;
                            break;

                        default:
                            return( MVI_CIP_BAD_INSTANCE );
                    } /* end switch( instance ) */

                    break;

                case MVI_CIP_SC_GET_MEMBER:
                case MVI_CIP_SC_SET_MEMBER:
                    /*
                    ** Not Yet Supported
                    */
                default:
                    return( MVI_CIP_BAD_SERVICE );
            } /* end switch( serviceCode ) */
            break;

        case MVI_CIP_IA_SIZE:

            /*
            ** Data size.
            ** Disallow anything but get attribute access.
            */

            /*
            ** Check the Service code.
            */
            switch( psServ->serviceCode )
            {
                case MVI_CIP_SC_SET_ATTR_SINGLE:
                    return( MVI_CIP_ATTR_NOT_SETTABLE );

                case MVI_CIP_SC_GET_ATTR_SINGLE:
                    /*
                    ** Return the size for the instance specified.
                    */
                    switch( psServ->instance )
                    {
                        case INPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&TxDatConnSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case OUTPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&RxDatConnSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case CONFIG_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&CfgDataSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case STATUS_IN_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&StsDataSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case MSGIN_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&MsgInDataSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case MSGOUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&MsgOutDataSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        default:
                            return( MVI_CIP_BAD_INSTANCE );
                    } /* end switch( instance ) */

                    break;

                case MVI_CIP_SC_GET_MEMBER:
                case MVI_CIP_SC_SET_MEMBER:
                    return( MVI_CIP_BAD_ATTR_DATA );

                default:
                    return( MVI_CIP_BAD_SERVICE );

            } /* end switch( serviceCode ) */
            break;

        case MVI_CIP_IA_NUM_MEMBERS:
        case MVI_CIP_IA_MEMBER_LIST:
        default:
            return( MVI_CIP_BAD_ATTR );

    } /* end switch( attribute ) */

    return( MVI_SUCCESS );

} /* end service_proc() */

/*
** Return Stack Checking to its default state
*/
#ifdef __BORLANDC__
#pragma option -N.
#endif
#ifdef _MSC_VER
#pragma check_stack ()
#endif
