/** $Revision:   1.0  $ **/
/**/
/************************************************************************
*
*   Title:  Sample barcode-reader application for the 1756-MVI
*
*   Abstract:
*
*   This sample program implements a simple serial interface to a barcode 
*   reader or similar device.  It uses the CIP API and transfers data
*   using unscheduled unconnected messagingi (CIP Generic instruction).
*
*   Note: This sample program performs the same function as the MVI API 
*   sample program BARCODE.C.  The same ladder code will work for either
*   program.
*
*
*   Environment:    1756-MVI Communications Module
*                   CIP API
*                   General Software DOS 6-XL
*                   Borland/Microsoft C/C++ Compiler (16-bit)
*
*   Author:         JMC
*                   Copyright (c) 1999-2000 Online Development, Inc.
*   
*
*   Receiving Serial Data
*   ---------------------
*
*   The first byte of the received message buffer contains the port
*   number from which the data was received (0=PRT1, 1=PRT2, 2=PRT3).
*
*   The termination character is transferred.  Unused bytes at
*   the end of the message buffer are set to zero.
*
*
*   Transmitting Serial Data
*   ------------------------
*
*   Messages received from the processor will be sent to the port
*   specified in the first byte of the message.
*
*
*   Using the program
*   -----------------
*
*   The program allows the baud rate and string termination character
*   to be specified on the command line.  It also allows either or
*   both serial ports to be specified.  For simplicity, the baud and
*   termination character parameters are applied to all serial ports.
*   The ports are setup for 8 bits, no parity, and 1 stop bit.
*
*   Syntax: 
*
*         barcode [-baud=index] [-tc=term char] -p[1|2|3]
*
*   Options:
*
*         -baud=[index]     Specifies serial port baud rate.  Default
*                           is 9600 (index=7).
*
*         -tc=[term char]   Specifies termination character.  ASCII
*                           value of character in decimal is expected.
*                           Default is 10 (line feed).
*
*   At least one of the following must be specified:
*
*         -p1               Indicates that PRT1 is enabled
*
*         -p2               Indicates that PRT2 is enabled
*
*         -p3               Indicates that PRT3 is enabled
*                   
* 
************************************************************************/


/*=======================================================================
=                           INCLUDE FILES                               =
=======================================================================*/

#include    <stdio.h>
#include    <stdlib.h>
#include    <conio.h>
#include    <string.h>
#include    "mvispapi.h"
#include    "cipapi.h"


/*=======================================================================
=                    MODULE WIDE DEFINITIONS                            =
=======================================================================*/

#define MAX_DATA_SZ       500   // Maximum string length that can be transferred

/*
 * Instance numbers implemented
 */
#define DATA_INPUT_INSTANCE      1
#define DATA_OUTPUT_INSTANCE     2
#define STATUS_INPUT_INSTANCE    5
#define STATUS_OUTPUT_INSTANCE   6
#define MSG_INPUT_INSTANCE       7
#define MSG_OUTPUT_INSTANCE      8

/*
 * Instance sizes
 */
#define DATA_INPUT_MAXSIZE     500      // max size in bytes
#define DATA_OUTPUT_MAXSIZE    500      // max size in bytes
#define STATUS_INPUT_MAXSIZE     6      // max size in bytes
#define STATUS_OUTPUT_MAXSIZE    0      // max size in bytes
#define MSG_OUTPUT_MAXSIZE     500      // max size in bytes
#define MSG_INPUT_MAXSIZE      500      // max size in bytes

// Serial Port Control Structure
typedef struct
{
    int  Enable;                // 1 means port is enabled for barcode data
    int  ComPort;               // Set to COM1, COM2, or COM3
    BYTE TermChar;              // termination character
    WORD StringLength;          // Max string length
    WORD InBufidx;              // Current receive data buffer pointer
    BYTE InBuf[MAX_DATA_SZ+1];  // Message buffer
    BYTE msgBusy;               // 1=message for controller is ready
} PORTCFG;


/*=======================================================================
=                      LOCAL SUPPORT PROTOTYPES                         =
=======================================================================*/

void ErrorExit(int);
void servicePort(PORTCFG *);
void checkTxMsg(void);


/*=======================================================================
=                    MODULE WIDE GLOBAL VARIABLES                       =
=======================================================================*/

MVIHANDLE handle;           // Handle returned from MVIcip_Open function
MVIHANDLE objHandle;        // Handle returned from MVIcip_RegisterAssemblyObj
PORTCFG Port1Cfg;           // Allocate data for port 1
PORTCFG Port2Cfg;           // Allocate data for port 2
PORTCFG Port3Cfg;           // Allocate data for port 3

// Data connection
MVIHANDLE dataConnHandle=0;                 // Handle for data connection
BYTE dataConnOpen=0;                        // 1 if data connection is open
WORD dataInputSize=0;                       // Set in connect_proc
WORD dataOutputSize=0;                      // Set in connect_proc
BYTE dataInputBuf[DATA_INPUT_MAXSIZE];      // Data input buffer
BYTE dataOutputBuf[DATA_OUTPUT_MAXSIZE];    // Data output buffer

// Status connection
MVIHANDLE statusConnHandle=0;               // Handle for status connection
BYTE statusConnOpen=0;                      // 1 if status connection is open
WORD statusInputSize=0;                     // Set in connect_proc
WORD statusOutputSize=0;                    // Set in connect_proc
BYTE statusInputBuf[STATUS_INPUT_MAXSIZE];  // Status input buffer

// Messages (Class 3 services)
BYTE msgInputBusy=0;                    // Set when message is written by app, cleared when processor reads
WORD msgInputSize=0;                    // Set by application
BYTE msgInputBuf[MSG_INPUT_MAXSIZE];    // Message input buffer
BYTE msgOutputBusy=0;                   // Set when message is received, cleared when app reads
WORD msgOutputSize=0;                   // Set in service_proc
BYTE msgOutputBuf[MSG_OUTPUT_MAXSIZE];  // Message output buffer


/*=======================================================================
=                       LOCAL SUPPORT ROUTINES                          =
=======================================================================*/

/************************************************************************
*
*     Entry point:                                                      
*       ErrorExit                                           
*
*     Description:                                                      
*       This routine is called whenever an error code is returned from
*       an API routine.  It displays the message associated with the
*       error, then exits the program.
*       
*     Arguments:                                                        
*
*       errcode             : int                           ( input )
*         Error return value from a backplane API function
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void ErrorExit(int errcode)
{
    char errbuf[80];

    if (errcode)
    {
        MVIcip_ErrorString(errcode, errbuf);   // get error message
        printf("%s\n", errbuf);
    }

    MVIcip_Close(handle);                      // should always close before exiting
    exit(1);
}


/************************************************************************
*
*     Entry point:                                                      
*       usage
*
*     Description:                                                      
*       This routine is called when invalid command-line options
*       are encountered.
*       
*     Arguments:                                                        
*
*       name                : char *                        ( input )
*         program name string
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void usage(char *name)
{
    printf("\nUsage: %s [-baud=index] [-tc=term ch] [-setup] -p[1|2|3]", name);
    printf("\nOptions:");
    printf("\n   -baud=index     index specifies baud rate (default=7 or 9600 baud)");
    printf("\n   -tc=term ch     termination character value in decimal (default=10)");
    printf("\n   -setup          do not exit if setup jumper installed");
    printf("\nAt least one of the following must be specified:");
    printf("\n   -p1             enable COM1");
    printf("\n   -p2             enable COM2");
    printf("\n   -p3             enable COM3");
}


/*
** Stack Checking must be disabled for callback routines !!
*/
#ifdef __BORLANDC__
#pragma option -N-
#endif
#ifdef _MSC_VER
#pragma check_stack (off)
#endif

/************************************************************************
*
*     Entry point:                                                      
*       connect_proc                                           
*
*     Description:                                                      
*       Class 1 connection callback routine.  This routine is called when
*       a forward open or close is received for the registered assembly
*       object.
*
*       The following Class 1 connection points (instances) are supported:
*
*       1 - Input data
*       2 - Output data
*       5 - Status input
*       6 - Status output (0 length)
*
*       Note: The Input data and Output data connections are not used by
*       this sample application.  They are supported here as illustration
*       and to maintain compatibility with the MVI API version of this
*       application.
*
*     Arguments:                                                        
*
*       objHandle           : MVIHANDLE                     ( input )
*         Object handle from MVIcip_RegisterAssemblyObj
*
*       psConn              : MVICIPCONNSTRUC *             ( input )
*         Pointer to a structure containing information from the
*         forward open/close.
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
#ifdef __BORLANDC__
#pragma warn -par*
#endif
MVICALLBACK connect_proc(
    MVIHANDLE objHandle,
    MVICIPCONNSTRUC *psConn )
{

    switch( psConn->reason )        // Open, Open Complete, or Close
    {
        case MVI_CIP_CONN_OPEN:                 // Open Request

            // Check for data connection
            if ( (psConn->producerCP == DATA_INPUT_INSTANCE) &&
                 (psConn->consumerCP == DATA_OUTPUT_INSTANCE) )
            {
                // Only one connection to be active at a time
                if( dataConnHandle != (MVIHANDLE) NULL )
                {
                    *psConn->extendederr = MVI_CIP_EX_CONNECTION_USED;
                    return( MVI_CIP_FAILURE );
                }

                // Check size
                if( (psConn->txDataSize > DATA_INPUT_MAXSIZE) ||
                    (psConn->rxDataSize > DATA_OUTPUT_MAXSIZE) )
                {
                   *psConn->extendederr = MVI_CIP_EX_BAD_SIZE;
                   return( MVI_CIP_FAILURE );
                }

                // Everything checks out - accept the connection
                // Save the connection handle and other info
                dataConnHandle = psConn->connHandle;
                dataInputSize = psConn->txDataSize;
                dataOutputSize = psConn->rxDataSize;
            }

            // Check for status Connection
            else if ( (psConn->producerCP == STATUS_INPUT_INSTANCE) &&
                      (psConn->consumerCP == STATUS_OUTPUT_INSTANCE) )
            {
                // Only one connection to be active at a time
                if( statusConnHandle != (MVIHANDLE) NULL )
                {
                    *psConn->extendederr = MVI_CIP_EX_CONNECTION_USED;
                    return( MVI_CIP_FAILURE );
                }

                // Check size
                if( (psConn->txDataSize > STATUS_INPUT_MAXSIZE) ||
                    (psConn->rxDataSize > STATUS_OUTPUT_MAXSIZE) )
                {
                    *psConn->extendederr = MVI_CIP_EX_BAD_SIZE;
                    return( MVI_CIP_FAILURE );
                }

                // Accept the connection
                // Save status connection handle
                statusConnHandle = psConn->connHandle;
                statusInputSize = psConn->txDataSize;

            }
            else    // Unknown connection point
            {
                return( MVI_CIP_FAILURE );
            }
            break;

        case MVI_CIP_CONN_OPEN_COMPLETE:        // Open Complete
            /*
            ** The connection is now up and running. If desired, load
            ** initial transmit data now. Also now is the time to notify
            ** the application threads that the connection is running.
            */
            if ( psConn->connHandle == dataConnHandle )
            {
                dataConnOpen = 1;          // indicate connection is now open
            }
            else if ( psConn->connHandle == statusConnHandle )
            {
                statusConnOpen = 1;        // indicate connection is now open
            }

            break;

        case MVI_CIP_CONN_CLOSE:                // Connection Closed
            /*
            ** The connection referenced by the connHandle has been
            ** closed.
            */
            if ( psConn->connHandle == dataConnHandle )
            {
                /*
                ** Here we need to:
                **    Free any allocated resources
                **    Clear handles
                */
                dataConnOpen = 0;           // indicate connection is now closed 
                dataConnHandle = (MVIHANDLE) NULL;
            }
            else if ( psConn->connHandle == statusConnHandle )
            {
                statusConnOpen = 0;         // indicate connection is now closed 
                statusConnHandle = (MVIHANDLE) NULL;
            }

            break;

        default:
            return( MVI_CIP_FAILURE );
    }

    return( MVI_SUCCESS );

} /* end connect_proc() */
#ifdef __BORLANDC__
#pragma warn .par*
#endif


/************************************************************************
*
*     Entry point:                                                      
*       service_proc                                           
*
*     Description:                                                      
*       Class 3 service callback routine.  This routine is called when
*       an unscheduled Class 3 service request is received.
*
*       This application supports the following Class 3 instances (called 
*       'Object ID' in the 5550 message configuration dialog)
*       of the assembly object (class code/object type 4):
*
*       1 - Input data (get only)
*       2 - Output data (get & set)
*       5 - Status input (get only)
*       7 - Message input data (get only)
*       8 - Message output data (get & set)
*
*       The following services are supported:
*
*       Get Attribute Single: 0x0E
*       Set Attribute Single: 0x10
*
*       The following attributes are supported:
*
*       Data: 0x03
*       Size: 0x04
*
*       There are two Class 3 messages that are used to implement 
*       MVI Messaging:
*
*           Send a message to the MVI module
*               - attribute 3 (data)
*               - instance 8 (msg output data)
*               - service 0x10 (set)
*
*           Read a message from the MVI module
*               - attribute 3 (data)
*               - instance 7 (msg input data)
*               - service 0x0E (get)
*
*     Arguments:                                                        
*
*       objHandle           : MVIHANDLE                     ( input )
*         Object handle from MVIcip_RegisterAssemblyObj
*
*       psServ              : MVICIPSERVSTRUC *             ( input )
*         Pointer to a structure containing information from the
*         service request.
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       MVI_SUCCESS
*       MVI_CIP_ATTR_NOT_SETTABLE
*       MVI_CIP_PARTIAL_DATA
*       MVI_CIP_BAD_SIZE
*       MVI_CIP_NO_RESOURCE 
*       MVI_CIP_BAD_SERVICE
*       MVI_CIP_BAD_INSTANCE 
*       MVI_CIP_BAD_ATTR_DATA
*       MVI_CIP_BAD_ATTR
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
#ifdef __BORLANDC__
#pragma warn -par*
#endif
MVICALLBACK service_proc(
    MVIHANDLE objHandle,
    MVICIPSERVSTRUC *psServ )
{

    /*
    ** Process according to the instance attribute.
    */
    switch( psServ->attribute )
    {
        case MVI_CIP_IA_DATA:

            /*
            ** Check the Service code.
            */
            switch( psServ->serviceCode )
            {
                case MVI_CIP_SC_SET_ATTR_SINGLE:

                    if( *psServ->msgSize != 0 )
                    {
                        switch( psServ->instance )
                        {
                            // Do not allow writes to any of these
                            case DATA_INPUT_INSTANCE:
                            case STATUS_INPUT_INSTANCE:
                            case STATUS_OUTPUT_INSTANCE:
                            case MSG_INPUT_INSTANCE:
                                return(MVI_CIP_ATTR_NOT_SETTABLE);

                            // Allow write to output data
                            case DATA_OUTPUT_INSTANCE:
                                if( *psServ->msgSize != dataOutputSize )
                                {
                                    return( MVI_CIP_PARTIAL_DATA );
                                }
                                memcpy(dataOutputBuf, *psServ->msgBuf, dataOutputSize);
                                break;

                            ///////////////////////////////////////////////////
                            // The useful one - a new message for the module
                            ///////////////////////////////////////////////////
                            case MSG_OUTPUT_INSTANCE:

                                if (msgOutputBusy)      // Make sure the last msg has been read
                                {
                                    return( MVI_CIP_DEVICE_BUSY );
                                }
                                if( *psServ->msgSize > MSG_OUTPUT_MAXSIZE )
                                {
                                    return( MVI_CIP_DATA_TOO_BIG );
                                }
                                msgOutputSize = *psServ->msgSize;
                                memcpy(msgOutputBuf, *psServ->msgBuf, msgOutputSize);
                                msgOutputBusy = 1;      // Notify application that a message is ready
                                break;

                            default:
                                return( MVI_CIP_BAD_INSTANCE );

                        } /* end switch( instance ) */
                    }
                    break;

                case MVI_CIP_SC_GET_ATTR_SINGLE:
                    /*
                    ** Return the data for the instance specified.
                    */
                    switch( psServ->instance )
                    {
                        case DATA_INPUT_INSTANCE:
                            *psServ->msgBuf   = dataInputBuf;
                            *psServ->msgSize  = dataInputSize;
                            break;

                        case DATA_OUTPUT_INSTANCE:
                            *psServ->msgBuf   = dataOutputBuf;
                            *psServ->msgSize  = dataOutputSize;
                            break;

                        case STATUS_INPUT_INSTANCE:
                            *psServ->msgBuf   = statusInputBuf;
                            *psServ->msgSize  = statusInputSize;
                            break;
                            
                        ///////////////////////////////////////////////////
                        // The useful one - Read a message from the module
                        ///////////////////////////////////////////////////
                        case MSG_INPUT_INSTANCE:
                            if ( msgInputBusy )     // Is there a message?
                            {
                                *psServ->msgBuf   = msgInputBuf;
                                *psServ->msgSize  = msgInputSize;
                                msgInputBusy = 0;   // Notify application that message has been read
                            }
                            else                    // Return error if no new message
                            {
                                return( MVI_CIP_DEVICE_BUSY );
                            }
                            break;

                        // Read back the last message written
                        case MSG_OUTPUT_INSTANCE:
                            *psServ->msgBuf   = msgOutputBuf;
                            *psServ->msgSize  = msgOutputSize;
                            break;

                        default:
                            return( MVI_CIP_BAD_INSTANCE );

                    } /* end switch( instance ) */

                    break;

                case MVI_CIP_SC_GET_MEMBER:
                case MVI_CIP_SC_SET_MEMBER:
                default:
                    return( MVI_CIP_BAD_SERVICE );

            } /* end switch( serviceCode ) */
            break;

        case MVI_CIP_IA_SIZE:

            /*
            ** Data size.
            ** Disallow anything but get attribute access.
            */

            /*
            ** Check the Service code.
            */
            switch( psServ->serviceCode )
            {
                case MVI_CIP_SC_SET_ATTR_SINGLE:
                    return( MVI_CIP_ATTR_NOT_SETTABLE );

                case MVI_CIP_SC_GET_ATTR_SINGLE:
                    /*
                    ** Return the size for the instance specified.
                    */
                    switch( psServ->instance )
                    {
                        case DATA_INPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&dataInputSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case DATA_OUTPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&dataOutputSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case STATUS_INPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&statusInputSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case STATUS_OUTPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&statusOutputSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case MSG_INPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&msgInputSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case MSG_OUTPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&msgOutputSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        default:
                            return( MVI_CIP_BAD_INSTANCE );
                    } /* end switch( instance ) */

                    break;

                case MVI_CIP_SC_GET_MEMBER:
                case MVI_CIP_SC_SET_MEMBER:
                    return( MVI_CIP_BAD_ATTR_DATA );

                default:
                    return( MVI_CIP_BAD_SERVICE );

            } /* end switch( serviceCode ) */
            break;

        case MVI_CIP_IA_NUM_MEMBERS:
        case MVI_CIP_IA_MEMBER_LIST:
        default:
            return( MVI_CIP_BAD_ATTR );

    } /* end switch( attribute ) */

    return( MVI_SUCCESS );

} /* end service_proc() */
#ifdef __BORLANDC__
#pragma warn .par*
#endif

/*
** Return Stack Checking to its default state
*/
#ifdef __BORLANDC__
#pragma option -N.
#endif
#ifdef _MSC_VER
#pragma check_stack ()
#endif


/*=======================================================================
=                       MAIN ENTRY POINT                                =
=======================================================================*/

/************************************************************************
*
*     Entry point:                                                      
*       main                                           
*
*     Description:                                                      
*
*     Arguments:                                                        
*       none
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       none
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
*     debugging/error printf's will only be seen if console is enabled.
*
************************************************************************/
void main(int argc, char *argv[])
{
    char *arg;
    int rc;
    int n;
    int mode;
    BYTE Baud = BAUD_9600;      // default baud rate
    BYTE TermChar = _LF;        // default termination character
    int Setup = 0;

    // init port config structs
    memset(&Port1Cfg, 0, sizeof(PORTCFG));
    memset(&Port2Cfg, 0, sizeof(PORTCFG));
    memset(&Port3Cfg, 0, sizeof(PORTCFG));
    Port1Cfg.InBufidx = 1;      // leave first byte for port number
    Port2Cfg.InBufidx = 1;      // leave first byte for port number
    Port3Cfg.InBufidx = 1;      // leave first byte for port number

    // parse the command line options
    for (n=1; n<argc; n++) {    // scan for arguments
        arg = argv[n];

		if (strnicmp("-baud=", arg, 6) == 0) {
            Baud = (BYTE) atoi(arg+6);
            if (Baud > BAUD_115200)
            {
                usage(argv[0]);
                exit(1);
            }
            continue;
        }

		if (strnicmp("-tc=", arg, 4) == 0) {
            TermChar = (BYTE) atoi(arg+4);
            continue;
        }

		if (strnicmp("-setup", arg, 6) == 0) {
            Setup = 1;
            continue;
        }

		if (strnicmp("-p1", arg, 3) == 0) {
            Port1Cfg.Enable = 1;
            continue;
        }

		if (strnicmp("-p2", arg, 3) == 0) {
            Port2Cfg.Enable = 1;
            continue;
        }

		if (strnicmp("-p3", arg, 3) == 0) {
            Port3Cfg.Enable = 1;
            continue;
        }

        usage(argv[0]);
        exit(1);
    }

    // Make sure at least one port is enabled
    if ((!Port1Cfg.Enable) && (!Port2Cfg.Enable) && (!Port3Cfg.Enable))
    {
        usage(argv[0]);
        exit(1);
    }

    // Open the Backplane API
    if (MVI_SUCCESS != (rc = MVIcip_Open(&handle)))
    {
        printf("\nMVIcip_Open failed: %d\n", rc);
        ErrorExit(rc);
    }

    // Check for setup mode - if so, exit now unless setup option is present.
    // The purpose for this is to allow the user to regain control of the
    // module by installing the Setup Jumper.  In a typical configuration,
    // the module application is run from AUTOEXEC.BAT.  In this case,
    // we exit the application if the Setup Jumper is installed, unless
    // the -setup command line option is preset.  This option is useful
    // for application debugging.
    MVIcip_GetSetupMode(handle, &mode);
    if (!Setup && mode)
    {   // Print the banner and version only if setup jumper is on
        printf("\n1756-MVI Barcode Sample Application (CIP API example)");
        printf("\nCopyright (c) 2000 Online Development, Inc.\n");
        printf("\nSetup jumper installed: exiting application\n");
        MVIcip_Close(handle);
        exit(0);
    }

    // Register the assembly object
    if (MVI_SUCCESS != (rc = MVIcip_RegisterAssemblyObj( handle, &objHandle,
                                0L, connect_proc, service_proc, NULL )))
    {
        printf("\nMVIcip_RegisterAssemblyObj failed: %d\n", rc);
        ErrorExit(rc);
    }

    // Initialize the serial port(s)
    if (Port1Cfg.Enable)
    {
        rc = MVIsp_Open(COM1, Baud, PARITY_NONE, WORDLEN8, STOPBITS1);
        if (rc != MVI_SUCCESS)
            ErrorExit(rc);
        Port1Cfg.ComPort = COM1;
        Port1Cfg.StringLength = MAX_DATA_SZ - 1;
        Port1Cfg.TermChar = TermChar;
    }
    if (Port2Cfg.Enable)
    {
        rc = MVIsp_Open(COM2, Baud, PARITY_NONE, WORDLEN8, STOPBITS1);
        if (rc != MVI_SUCCESS)
            ErrorExit(rc);
        Port2Cfg.ComPort = COM2;
        Port2Cfg.StringLength = MAX_DATA_SZ - 1;
        Port2Cfg.TermChar = TermChar;
    }
    if (Port3Cfg.Enable)
    {
        rc = MVIsp_Open(COM3, Baud, PARITY_NONE, WORDLEN8, STOPBITS1);
        if (rc != MVI_SUCCESS)
            ErrorExit(rc);
        Port3Cfg.ComPort = COM3;
        Port3Cfg.StringLength = MAX_DATA_SZ - 1;
        Port3Cfg.TermChar = TermChar;
    }

    // Main Loop - never exits
    while(1)
    {
        servicePort(&Port1Cfg);     // service rx data
        servicePort(&Port2Cfg);     // service rx data
        servicePort(&Port3Cfg);     // service rx data
        checkTxMsg();               // Look for message to transmit
    }

    // The following line is needed if the loop above is allowed to exit.
//    MVIcip_Close(handle);

}


/************************************************************************
*
*     Entry point:                                                      
*       servicePort                                           
*
*     Description:                                                      
*       Check for data from the serial port.  If data is ready, move
*       it to the port's buffer.  If the message is complete (either
*       by matching the termination character or the string length),
*       then send the data to the PLC.
*
*     Arguments:                                                        
*
*       portcfg             : PORTCFG *                     ( input )
*         Pointer to port control structure
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void servicePort(PORTCFG *portcfg)
{
    int rc;
    int len;
    int timeout;

    // Return if port is not enabled 
    if (!portcfg->Enable)
        return;

    // Calculate number of characters left to go
    len = portcfg->StringLength - portcfg->InBufidx;

    rc = MVIsp_Gets(portcfg->ComPort, &portcfg->InBuf[portcfg->InBufidx],
             portcfg->TermChar, &len, TIMEOUT_ASAP);

    if (len == 0)                       // Return if no data in queue
        return;

    // Update buffer index
    portcfg->InBufidx += len;

    // Termination test - see if string is complete
    if (rc == MVI_SUCCESS)
    {
        // String is complete - build a message and send to PLC 

        // Write port number to buffer
        portcfg->InBuf[0] = portcfg->ComPort;

        // Write data for controller to fetch - blink LED for debugging
        MVIcip_SetUserLED(handle, MVI_LED_USER1, MVI_LED_STATE_ON);

        memcpy(msgInputBuf, portcfg->InBuf, portcfg->InBufidx); // copy data to message buffer
        msgInputSize = portcfg->InBufidx;
        msgInputBusy = 1;                   // allow controller to read message

        // Wait for controller to read message - if timeout, data may be lost
        timeout = 200;                  // timeout in approx 2 seconds
        while(msgInputBusy && timeout--)
        {
            MVIcip_Sleep(handle, 10);   // Sleep approx 10ms
        }
        MVIcip_SetUserLED(handle, MVI_LED_USER1, MVI_LED_STATE_OFF);

        portcfg->InBufidx = 1;      // Ready for new serial data

        // zero data buffer to get ready for next message
        memset(&portcfg->InBuf, 0, sizeof(portcfg->InBuf));
    }
}


/************************************************************************
*
*     Entry point:                                                      
*       checkTxMsg
*
*     Description:                                                      
*       Check for a message from the PLC to transmit.  If there is
*       a message, send it to the appropriate port, as indicated
*       by the first byte of the message.
*
*     Arguments:                                                        
*       None
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void checkTxMsg(void)
{
    int len;

    if (msgOutputBusy)          // did we get a message?
    {
        // Blink the LED for debugging
        MVIcip_SetUserLED(handle, MVI_LED_USER2, MVI_LED_STATE_ON);

        len = msgOutputSize - 1;
        switch(msgOutputBuf[0]) // which port?
        {
            case 0:     // PRT1
                MVIsp_PutData(COM1, &msgOutputBuf[1], &len, 1000);
                break;

            case 1:     // PRT2
                MVIsp_PutData(COM2, &msgOutputBuf[1], &len, 1000);
                break;

            case 2:     // PRT3
                MVIsp_PutData(COM3, &msgOutputBuf[1], &len, 1000);
                break;

            default:    // Invalid port number - just ignore
                break;
        }

        MVIcip_SetUserLED(handle, MVI_LED_USER2, MVI_LED_STATE_OFF);
        msgOutputBusy = 0;      // ready for next message

    }
}


