/////////////////////////////////////////////////////////////////////////////
// backplane_sample.c
//
// This file contains sample code for the MVI69E-LDM module.
//
// (c) 2013, Prosoft Technology , All Rights Reserved
//
// MODIFIED :
//  4/18/2013 -- PJB
//	 3/31/2017 -- JAC	→ Change 'if' statement to 'while' statement
// 							  during open access to backplane/hardware driver
//							  in which module waits for a connection with Processor 
/////////////////////////////////////////////////////////////////////////////


#include <stdlib.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>

#include <stdio.h>
#include <string.h>
#include <signal.h>

#include "mvi69-types-and-variables.h"

volatile int			Last_Run_Mode =  -1;
int								run_mode = 16;

// timeout to wait for scan (in milliseconds )
#define TIMEOUT  1000L

DWORD			scan_counter;
volatile int	Done = FALSE;
volatile int exit_code = 0;
volatile int PLC_DB_Initialized = FALSE;

int tw_ldm_is_connected_to_plc(void)
{
	return (PLC_DB_Initialized &&  (Last_Run_Mode == MVI69_RUN_MODE)) ? 1 : 0;
}

void stop_tw_and_exit(void)
{
	int result;
	exit_code = 0;

	tw_ldm_log(TW_LDM_LOG_INFO, "Disconnecting from ThingWorx Server.");
	result = tw_ldm_disconnect();
	tw_ldm_log(TW_LDM_LOG_NOTICE, "Disconnected from ThingWorx Server, result = %d", result);

	tw_ldm_log(TW_LDM_LOG_NOTICE, "Backplane is closed");

	tw_ldm_log(TW_LDM_LOG_NOTICE, "Stopping ThingWorx subsystem.");
	result = tw_ldm_clean();
	tw_ldm_log(TW_LDM_LOG_NOTICE, "Stopped ThingWorx subsystem, result = %d", result);

	tw_ldm_free_config(&gl_app_config);

	exit(exit_code);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//	sigquit_handler( int sig )
//
//   sig - is the signal that caused the invocation.
//
// This is the normal exit routine for the application.  It is called when the program exits and
// when the following signals are received: SIGSEGV, SIGQUIT, SIGINT or SIGTERM.  This function will cause
// the module to exit(not reboot) after the application closes all the resources.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
void sigquit_handler(int sig)
{
	const char* reason;
	switch (sig)
	{
	case SIGTERM:
		exit_code = 0;
		reason = "Termination";
		break;
	case SIGKILL:
		exit_code = 0;
		reason = "Kill , unblockable";
		break;
	case SIGQUIT:
		exit_code = 0;
		reason = "Quit";
		break;
	case SIGINT:
		exit_code = 0;
		reason = "Interrupt";
		break;
	default:
		exit_code = -2;
		reason = "unknown";
		break;
	}

	tw_ldm_log(TW_LDM_LOG_NOTICE, "\nReceived signal # %d [%s], scan counter = %d", sig, reason, scan_counter);

	// Call function stop_tw_and_exit only once, to prevent re-entrances due to exceptions during stopping.
	if (!Done)
	{
		Done = TRUE;
		MVI69_Close(MVI69_Handle);
		PLC_DB_Initialized = 0;
		stop_tw_and_exit();
	}
	else
		exit(-3);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//  main program
//			Input:
//				argc - argument count.
//				argv - list of arguments.
//
//			Description:  This program
//								- establishes communication with the Control Logix Controller.
//								- reads the Controller's Clock and displays
//								- Reads a block of data from the controller,
//											and writes that block back to the controller.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

int main( int argc, char * argv[] )
{
	int				result;
	char			buffer[256];
	char*     config_file_name = "config.json";

	uint32_t						last_tick = 0;
	uint32_t            last_status_print = 0;
	
	char pre_is_connected = -1;
	char pre_is_connecting = -1;

	memset(buffer, 0, sizeof(buffer));

	printf("\nProgram Initialization... Version 1.0.0.5\n");

	// set up special event handlers.

	signal( SIGSEGV, sigquit_handler );
	signal( SIGQUIT, sigquit_handler );
	signal( SIGINT,  sigquit_handler );
	signal( SIGTERM, sigquit_handler );
	
	if (argc > 1)
	{
		if (argv[1][0] == '-' && (argv[1][1] == 'h' || argv[1][1] == 'H'))
		{
			printf("\n Usage:\n tw-ldm-sample-app-mvi69 [config-file-name]\n  Communicates with ThingWorx Server and ControlLogix processor,");
			printf("\n maps thing properties to PLC tags and synchronizes their values.");
			printf("\n Optional parameter config-file-name: name of configuration file, by default \"config.json\"\n\n");

			exit(-1);
		}
		else
			config_file_name = argv[1];
	}

	// Load configuration file.
	printf("Loading configuration settings from file [%s]\n", config_file_name);
	result = tw_ldm_initialize(config_file_name, NULL);
	if (result != 0)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Failed to initialize ThingWorx subsystem, exiting");
		exit(result);
	}

  // main task loop, stay here forever
	while (!Done)
	{
		uint32_t current_tick;
		char is_connected;
		char is_connecting;

		if (!PLC_DB_Initialized)
		{
			//  open access to backplane and hardware
			result = open_backplane();
			if (result != 0)
			{
				tw_ldm_log(TW_LDM_LOG_ERROR, "Failed to open connection with backplane, error code = %d.", result);
			}
			else
			{
				// initialize LED
				MVI69_SetLED(MVI69_Handle, MVI69_LEDID_OK, MVI69_LED_STATE_GREEN);
				MVI69_SetLED(MVI69_Handle, MVI69_LEDID_CFG, MVI69_LED_STATE_GREEN);

				MVI69_SetLED(MVI69_Handle, MVI69_LEDID_P1, MVI69_LED_STATE_OFF);
				MVI69_SetLED(MVI69_Handle, MVI69_LEDID_P2, MVI69_LED_STATE_OFF);

				MVI69_SetLED(MVI69_Handle, MVI69_LEDID_BP, MVI69_LED_STATE_OFF);
				MVI69_SetLED(MVI69_Handle, MVI69_LEDID_NET, MVI69_LED_STATE_GREEN);

				tw_ldm_log(TW_LDM_LOG_INFO, "Backplane connected....");
			}
		}

		current_tick = get_tick_count();
		if ((current_tick - last_tick) > 1000)
		{
			last_tick = current_tick;

			is_connected = tw_ldm_is_connected();
			is_connecting = tw_ldm_is_connecting();
			if ((pre_is_connected != is_connected) || (pre_is_connecting != is_connecting) || ((current_tick - last_status_print) > (gl_app_config.status_print_interval * 1000)))
			{
				last_status_print = current_tick;
				tw_ldm_log(TW_LDM_LOG_NOTICE, "ThingWorx Server status: %s%s. Scan counter = %d", (is_connected) ? "connected" : " disconnected", (is_connecting ? ", connection in progress" : ""), scan_counter);
				pre_is_connected = is_connected;
				pre_is_connecting = is_connecting;
			}

			if (!is_connected && !is_connecting)
			{
				tw_ldm_log(TW_LDM_LOG_NOTICE, "Trying to connect to ThingWorx Server");
				tw_ldm_disconnect();
				usleep(1000000);
				tw_ldm_connect();
			}
		}

		if (PLC_DB_Initialized)
		{
			// read the scan mode from the backplane.
			result = MVI69_GetScanMode(MVI69_Handle, &run_mode);
			if (result != MVI69_SUCCESS)
			{
				tw_ldm_log(TW_LDM_LOG_ERROR, "MVI69 Get Scan Mode failed: %d", result);
			}

			if (run_mode != Last_Run_Mode)
			{
				tw_ldm_log(TW_LDM_LOG_INFO, "Run mode changed from %s to %s", MODE(Last_Run_Mode), MODE(run_mode));
				Last_Run_Mode = run_mode;
			}

			// wait for the the next input (output does exactly the same thing) scan.
			result = MVI69_WaitForInputScan(MVI69_Handle, TIMEOUT);
			//		result = MVI69_WaitForOutputScan( MVI69_Handle, TIMEOUT );

			if (result != MVI69_ERR_TIMEOUT)
			{
				if (result != MVI69_SUCCESS)
				{
					printf("MVI69 wait for scan failed: %d\n", result);
				}

				// read the current scan count
				result = MVI69_GetScanCounter(MVI69_Handle, &scan_counter);
				if (result != MVI69_SUCCESS)
				{
					tw_ldm_log(TW_LDM_LOG_ERROR, "MVI69 Get Scan Counter failed: %d", result);
				}
			}
		}
		usleep(1000);
	};

	return 0;
}






