/////////////////////////////////////////////////////////////////////////////
//
// ModuleInfo.c
//
// Runtime module info access functions
//
// Copyright (c) 2011, ProSoft Technology, All Rights Reserved
//
//
// DATE     : Aug/09/2011
// MODIFIED :
// 08/09/2011 HYU - Created
// 08/19/2011 HYU - Initialized the share memory ID with -1 instead of 0.
// 10/31/2011 HYU - Added GetLinuxVersion() and GetFirmwareData().
// 11/02/2011 HYU - Made the thread-safe functions by using semaphore.
// 11/04/2011 HYU - Modified GetFirmwareData to get the product info from the firmwaredata file.
// 11/29/2011 HYU - Used semget instead of sem_open because sem_open will be failed with 'Function not implemented'
//                  if /tmp is not mounted to tmpfs by configuring the /etc/fstab file.
// 12/01/2011 HYU - Defined USE_NAME_SEM to use sem_open because tmpfs will be mounted to /tmp.
// 01/04/2012 HYU - Added the Scan Count into MODULE_INFO.
// 02/23/2012 HYU - Modified MODULE_INFO to make Module Status Data aligned on the word boundaries.
// 03/31/2012 HYU - Modified InitModuleInfoRes() to fill netInfo by the order of eth0,eth1.
// 08/30/2012 HYU - Changed for wireless.
//
/////////////////////////////////////////////////////////////////////////////

#define USE_NAME_SEM

#include <unistd.h>
#include <sys/shm.h>
#include <sys/msg.h>
#include <string.h>
#include <stdio.h>
#include <ifaddrs.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#ifdef USE_NAME_SEM
#include <semaphore.h>
#else
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#endif

#include "ocxbpapi.h"
#include "moduleinfo.h"
#include "nethelper.h"




#define PROD_IPC_KEY   ( 'M' << 24 | '5' << 16 | '6' << 8 | '3' )
#define RWALL          0666

#ifndef offsetof
#define offsetof(s,m)   (size_t)&(((s *)0)->m)
#endif

#define   INVALID_SHM_ID           -1


static void GetLinuxVersion( char * OSVersion, char * OSBuild );
static void GetFirmwareData( MODULE_INFO * moduleInfo );


static struct
{
    char * name;
    int offset;
    int max_size;
} g_infoTable[] = {
    { "PRODUCT_TYPE", offsetof(MODULE_INFO, ProductType), MAX_NAME_LEN-1 },
    { "PRODUCT_NAME", offsetof(MODULE_INFO, ProductName), MAX_NAME_LEN-1 },
    { "PRODUCT_CODE", offsetof(MODULE_INFO, ProductCode), MAX_MODULE_CODE_SIZE-1 },
    { "FIRMWARE_VERSION", offsetof(MODULE_INFO, ProductRev), MAX_VERSION_INFO_SIZE-1 },
    { "FIRMWARE_DATE", offsetof(MODULE_INFO, FirmwareDate), MAX_DATETIME_SIZE-1 },
};


static int shmID = INVALID_SHM_ID;
static MODULE_INFO *g_pModuleInfo = NULL;


#ifdef USE_NAME_SEM

static char 	SEM_NAME[]= "MVIModuleInfoMutex";
static sem_t * 	g_mutex = NULL;

#define LOCK()   sem_wait(g_mutex)
#define UNLOCK() sem_post(g_mutex)
#define LOCK_WAIT(ts) sem_timedwait(g_mutex, &ts)

#else

static int g_semID = -1;

// This needs to be defined for semaphore operations
union semun
{
    int val;					//<= value for SETVAL
    struct semid_ds *buf;		//<= buffer for IPC_STAT & IPC_SET
    unsigned short int *array;	//<= array for GETALL & SETALL
    struct seminfo *__buf;		//<= buffer for IPC_INFO
};

#define SEM_READY		0

struct sembuf semLockReady     =	{SEM_READY, -1, SEM_UNDO};
struct sembuf semUnlockReady   =	{SEM_READY,  1, SEM_UNDO};

#define LOCK()     	semop(g_semID, &semLockReady, 1)
#define UNLOCK()   	semop(g_semID, &semUnlockReady, 1)
#define LOCK_WAIT(ts) semtimedop(g_semID, &semLockReady, 1, &ts)

#endif


////////////////////////////////////////////////////////////////////////////////////

static MODULE_INFO* i_GetModuleInfo( void )
{
    if ( shmID == INVALID_SHM_ID || g_pModuleInfo == NULL )
    {
        shmID = shmget( PROD_IPC_KEY, sizeof( MODULE_INFO ), RWALL );
        
        if ( shmID != INVALID_SHM_ID )
        {
#ifdef USE_NAME_SEM
            //create & initialize semaphore
            g_mutex = sem_open(SEM_NAME,O_CREAT,RWALL,1);
            if (g_mutex == NULL)
            {
                perror("sem_open failed");
                return NULL;
            }
#else
            g_semID = semget(PROD_IPC_KEY, 1, RWALL | IPC_CREAT);
#endif
            g_pModuleInfo = (MODULE_INFO *) shmat( shmID, NULL, 0 );
        }
    }
    
    return g_pModuleInfo;   /* already got it */
}

////////////////////////////////////////////////////////////////

int GetModuleInfo( MODULE_INFO * mi )
{
    MODULE_INFO *mod_info_ptr = i_GetModuleInfo();


    if ( mod_info_ptr == NULL )
    {
        memset( mi, 0, sizeof(MODULE_INFO));
        return -1;
    }

    struct timespec ts;
    clock_gettime(CLOCK_REALTIME, &ts);
    ts.tv_sec += 1;

    if ( LOCK_WAIT(ts) == -1 )
    {
        memset( mi, 0, sizeof( MODULE_INFO ));
        return -1;
    };

    memcpy( mi, mod_info_ptr, sizeof(MODULE_INFO));

    UNLOCK();

    return 0;
}

/////////////////////////////////////////////////////////////

int ReleaseModuleInfo( void )
{
#ifdef USE_NAME_SEM
    sem_close( g_mutex );
#else
    /* remove Synchronization Semaphore */
    if (semctl( g_semID, 0, IPC_RMID ) == -1)
    {
        perror( "Error Removing Semaphore" );
    }
#endif

    if ( shmID != INVALID_SHM_ID && g_pModuleInfo )
    {
        shmdt( g_pModuleInfo );       /* when everyone detaches, it will */
        shmID = INVALID_SHM_ID;        /* go away, I hope... */
        g_pModuleInfo = NULL;
        return 0;
    }
    
    return -1;
}

/////////////////////////////////////////////////////////////////////////

int CreateModuleInfo( void )
{
    if ( shmID == INVALID_SHM_ID || g_pModuleInfo == NULL )
    {
        shmID = shmget( PROD_IPC_KEY, sizeof( MODULE_INFO ), RWALL | IPC_CREAT );
        if ( shmID != INVALID_SHM_ID )
        {
#ifdef USE_NAME_SEM
            sem_unlink(SEM_NAME);
            //create & initialize semaphore
            g_mutex = sem_open(SEM_NAME,O_CREAT,RWALL,1);
            if (g_mutex == NULL)
            {
                perror("sem_open failed");
                return -1;
            }
#else
            g_semID = semget(PROD_IPC_KEY, 1, RWALL | IPC_CREAT);

            // Init the Synchronization Semaphore
            union semun sem_info;
            sem_info.val = 1;
            if (semctl(g_semID, 0, SETVAL, sem_info) == -1)
            {
                perror("semctl failed");
            }
#endif

            g_pModuleInfo = (MODULE_INFO *) shmat( shmID, NULL, 0 );
            if ( g_pModuleInfo )
            {
                LOCK();
                memset( g_pModuleInfo, 0, sizeof( MODULE_INFO ) );
                UNLOCK();
                return 0;
            }
        }
    }
    
    perror("Failed to create a product info share memory.");
    return -1;
}

//////////////////////////////////////////////////////////////////////////

int RegisterModuleInfo( void )
{
    if ( CreateModuleInfo() == 0 )
    {
        MODULE_INFO * mod_info_ptr = i_GetModuleInfo();

        LOCK();

        GetLinuxVersion( mod_info_ptr->OpRev, mod_info_ptr->OpRun );
        GetFirmwareData( mod_info_ptr );

        UNLOCK();
        
        return 0;
    }
    
    return -1;
}

/* Depricated on Datm3, the serial number is read only ( set at manufacturing time

int SetModuleSerialNo(unsigned long SerialNo)
{
	MODULE_INFO *mod_info_ptr = i_GetModuleInfo();
    if (mod_info_ptr == NULL)
    {
        return -1;
    }

    LOCK();
    
    mod_info_ptr->SerialNo = SerialNo;

    UNLOCK();
    
    return 0;
}
*/

/////////////////////////////////////////////////////////////

int SetModuleName(const char * moduleName)
{
	MODULE_INFO *mod_info_ptr = i_GetModuleInfo();
    if (mod_info_ptr == NULL)
    {
        return -1;
    }

    LOCK();
    
    strncpy(mod_info_ptr->ModuleName, moduleName, MAX_NAME_LEN-1);
    mod_info_ptr->ModuleName[MAX_NAME_LEN-1] = '\0';

    UNLOCK();
    
    return 0;
}

////////////////////////////////////////////////////////////

const char * GetModuleName()
{
    const char * pszModuleName = "NOT CONFIGURED";

    MODULE_INFO *mod_info_ptr = i_GetModuleInfo();
    if (mod_info_ptr != NULL)
    {
        pszModuleName = mod_info_ptr->ModuleName;
    }
    
    return pszModuleName;
}

////////////////////////////////////////////////////////////

int SetModuleRunningStatus(const char * status)
{
	MODULE_INFO *mod_info_ptr = i_GetModuleInfo();
    if (mod_info_ptr == NULL)
    {
        return -1;
    }
    
    LOCK();

    strncpy(mod_info_ptr->Status, status, MAX_NAME_LEN-1);
    mod_info_ptr->Status[MAX_NAME_LEN-1] = '\0';

    UNLOCK();

    return 0;
}

////////////////////////////////////////////////////////////

void IncModuleScanCount( void )
{
    MODULE_INFO *mod_info_ptr = i_GetModuleInfo();

    if (mod_info_ptr != NULL)
    {
        (mod_info_ptr->scan_count)++;
    }
}

/////////////////////////////////////////////////////////////

unsigned long GetModuleScanCount( void )
{
    MODULE_INFO *mod_info_ptr = i_GetModuleInfo();

    if (mod_info_ptr == NULL)
    {
        return 0;
    }

    return mod_info_ptr->scan_count;
}

/////////////////////////////////////////////////////////////

unsigned short * GetModuleStatusData( void )
{
    MODULE_INFO *mod_info_ptr = i_GetModuleInfo();

    if ( mod_info_ptr == NULL )
    {
        return NULL;
    }
    
    return (unsigned short *) &(mod_info_ptr->scan_count);
}

/////////////////////////////////////////////////////////////////////


int InitModuleInfoRes( OCXHANDLE ocx_handle, WORD vender_id, WORD device_type, WORD product_numeric_code )
{
	struct in_addr 		ip;        					// eth0 IP address
	struct in_addr 		netmask;  				 	// eth0 Network Mask
	struct in_addr 		loAddr;    					// localhost address
	unsigned char 		macAddr[MAX_MAC_SIZE];
	struct ifaddrs *	ifaddr;
	struct ifaddrs *	ifa;
	int 				family;
	int					revMajor;
	int					revMinor;
	int					result;
	int 				i 				= 0;
	MODULE_INFO* 		mod_info_ptr 	= i_GetModuleInfo();
	OCXCIPIDOBJ     	id_obj;


	if ( mod_info_ptr != NULL )
	{
		LOCK();

		// initialize the Module identity object

		result = OCXcip_GetIdObject( ocx_handle, &id_obj );
		if ( result != OCX_SUCCESS )
		{
			printf("\nError 0x%X getting the ID object\n", result );
		}

		// set object identification

		id_obj.VendorID = vender_id;
		id_obj.DeviceType = device_type;
		id_obj.ProductCode = product_numeric_code;

		sscanf(mod_info_ptr->ProductRev, "%d.%d", &revMajor, &revMinor);
		id_obj.MajorRevision = revMajor;
		id_obj.MinorRevision = revMinor;

		strcpy((char *)id_obj.Name, mod_info_ptr->ModuleName );

		result = OCXcip_SetIdObject( ocx_handle, &id_obj );
		if ( result != OCX_SUCCESS )
		{
			printf("\nError 0x%X setting the ID object\n", result );
		}

		// transfer id info to mod_info structure
		mod_info_ptr->SerialNo 	= id_obj.SerialNo;
		mod_info_ptr->Slot 		= id_obj.Slot;

		if ( getifaddrs( &ifaddr ) == -1 )
		{
			printf("\nError getifaddrs in InitModuleInfoRes\n");
		}
		else
		{
			loAddr.s_addr = htonl(INADDR_LOOPBACK);

			/* Walk through linked list, maintaining head pointer so we can free list later */
			for ( ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next)
			{
				if ( ifa->ifa_addr != NULL )
				{

					family = ifa->ifa_addr->sa_family;
					if (family == AF_INET)
					{
						if ( getMACAddress(ifa->ifa_name, macAddr) != 0)
						{
							// ignore the entry
							continue;
						}

						ip = ((struct sockaddr_in *)ifa->ifa_addr)->sin_addr;

						if (loAddr.s_addr == ip.s_addr || strchr(ifa->ifa_name, ':') != NULL)
						{
							// ignore the localhost entry
							continue;
						}

						if (strcmp(ifa->ifa_name, ENET_IF0_NAME) == 0)
						{
							i = ENET_INFO_IF0;
						}
						else if (strcmp(ifa->ifa_name, ENET_IF1_NAME) == 0)
						{
							i = ENET_INFO_IF1;
						}
						else
						{
							i = ENET_INFO_WL0;
						}

						netmask = ((struct sockaddr_in *)ifa->ifa_netmask)->sin_addr;

						strncpy( mod_info_ptr->netInfo[i].id, ifa->ifa_name, MAX_RINFO_SIZE );

						strncpy( mod_info_ptr->netInfo[i].ip, inet_ntoa(ip), MAX_RINFO_SIZE );
						strncpy( mod_info_ptr->netInfo[i].netmask, inet_ntoa(netmask), MAX_RINFO_SIZE );

						memcpy( mod_info_ptr->netInfo[i].macAddr, macAddr, sizeof( macAddr ) );
					}
				}
			}

			freeifaddrs( ifaddr );
		}

		UNLOCK();
	}

	return i;
}

/////////////////////////////////////////////////////////////////////////////////////////////

void ShowModuleInfo( void )
{
    int i;
    
    MODULE_INFO mInfo;
    if (GetModuleInfo(&mInfo) < 0)
    {
    	printf("Failed to get the product info from share memory.\n");
    	return;
    }

    puts("MVI Module Info");
    puts("---------------------------");
    printf("Module Name: %s\n", mInfo.ModuleName);
    printf("Serial NO: %lX\n", mInfo.SerialNo);
    printf("Product Type: %s\n", mInfo.ProductType);
    printf("Product Name: %s\n", mInfo.ProductName);
    printf("Product Version: %s\n", mInfo.ProductRev);
    printf("Firmware Date: %s\n", mInfo.FirmwareDate);
    printf("Running Status: %s\n", mInfo.Status);
    for (i = 0; i < MAX_ENET_RESOURCE; i++)
    {
        printf("%s: %s %s %s %02X:%02X:%02X:%02X:%02X:%02X\n",
            mInfo.netInfo[i].id,
            mInfo.netInfo[i].ip,
            mInfo.netInfo[i].netmask,
            mInfo.netInfo[i].gateway,
            mInfo.netInfo[i].macAddr[5],
            mInfo.netInfo[i].macAddr[4],
            mInfo.netInfo[i].macAddr[3],
            mInfo.netInfo[i].macAddr[2],
            mInfo.netInfo[i].macAddr[1],
            mInfo.netInfo[i].macAddr[0]);
    }
    
}

////////////////////////////////////////////////////////////////////////////

static void GetLinuxVersion(char * OSVersion, char * OSBuild)
{
    char versionString[301];
    
    char linuxStr[20];
    char versionStr[20];
    char version[20];
    char *pBuild;
    char *pBuildEnd;
    
    // Open the /proc/version to get the version information
    FILE *procVersionFile = fopen("/proc/version","r");
    // Return if the file cannot be opened
    if (procVersionFile==NULL)
        return;

    fgets(versionString,300,procVersionFile);
    // Parse the linux version number
    sscanf(versionString,"%s %s %s",linuxStr,versionStr,version);
    // Parse the Build Number
    pBuild = strstr(versionString,"#");
    pBuildEnd = strstr(pBuild," ");
    *pBuildEnd = 0;

    strcpy(OSVersion, version);
    strcpy(OSBuild, pBuild);
    
    fclose(procVersionFile);
}

/////////////////////////////////////////////////////////
// read firmware info from /psft/firmwaredata file.
//  info include product name, type, code, version and date
/////////////////////////////////////////////////////////

static void GetFirmwareData( MODULE_INFO * moduleInfo )
{
    int 		i;
    const int 	size = sizeof(g_infoTable) / sizeof(g_infoTable[0]);
    char 		currentLine[301];
    char *		pos;
    char *		psz = (char *)moduleInfo;

    // make sure module info structure is valid
    if ( psz == NULL )
    {
        return;
    }
    
    // Open the /psft/firmwaredata file to get the version information of the firmware file
    FILE *firmwareVersionFile = fopen("/psft/firmwaredata","r");
    // exit if the file cannot be opened
    if ( firmwareVersionFile == NULL )
        return;

    while ( fgets(currentLine, 300, firmwareVersionFile ) != NULL )
    {
        pos = strchr(currentLine,'\r');
        if ( pos != NULL ) *pos = 0;

        pos = strchr(currentLine,'\n');
        if ( pos != NULL ) *pos = 0;

        pos = strchr(currentLine,':');
        if ( pos != NULL )
        {
            *pos = 0;
            pos++; // skip colon
        }
        else
        {
            continue;
        }

        // see if this line contain one of the pieces of information that we want
        for ( i = 0; i < size; i++ )
        {
            if ( strcmp(currentLine, g_infoTable[i].name) == 0 )
            {
            	// information element found, copy it into the module info structure
                strncpy( psz + g_infoTable[i].offset, pos, g_infoTable[i].max_size );
                break;
            }
        }
    }

    fclose( firmwareVersionFile );
}


