/**************************************************************
    Copyright (c) 2008, ProSoft Technology, All Rights Reserved

  File: nethelper.c

  Title:          Helper network functions for Linux

  Comments:       This simple library allows to read/write the MAC, IP address
                  from a specific network device in Linux.

  MODIFIED:
  12/03/2008 RAR Altered code slightly for return values and formatting.
  12/11/2008 DLK Modified to not display error if broadcast route already exists.
  03/09/2010 KDH Reversed digit order in getMACAddress.
  11/01/2011 HYU Added addMulticastRoute.
  12/05/2012 HYU Added resolve_ip and gethostip.
****************************************************************/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <linux/route.h>
#include <linux/sockios.h>
#include <linux/ethtool.h>
#include <linux/netdevice.h>
#include "nethelper.h"



/* /proc/net/dev */
#define PROCNETDEV "/proc/net/dev"



/*************************************************************************************************************
/ Purpose: Function that allows to read the IP address of a network interface
/
/ Params: char *ifname: Interface name that we want the information from. "eth0"
/              struct in_addr *ip: Buffer to hold the IP address
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int getIPAddress(char *ifname, struct in_addr *ip)
{
	struct ifreq reqIf;
	int sockfd;

	assert(ifname != NULL);
	assert(ip != NULL);

	// Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
		memset(ip, 0, sizeof(struct in_addr));
 		return NETHELPER_EXIT_FAILURE;
 	}

	// Copy the interface name
	strncpy(reqIf.ifr_name, ifname, sizeof(reqIf.ifr_name));
	// Get the IP address via an ioctl
	if(ioctl(sockfd, SIOCGIFADDR , &reqIf) < 0)
	{
		perror("ioctl");
		memset(ip, 0, sizeof(struct in_addr));
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}
	else
	{
		memcpy(ip,&(((struct sockaddr_in *)&reqIf.ifr_addr)->sin_addr),sizeof(struct in_addr));
		close(sockfd);
		return NETHELPER_EXIT_SUCCESS;
	}
}


/*************************************************************************************************************
/ Purpose: Function that allows to set the IP address of a network interface
/
/ Params: char *ifname: Interface name that we want the information from. "eth0"
/              struct in_addr *ip: Desired new IP Address
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int setIPAddress(char *ifname, struct in_addr *ip)
{
	struct ifreq reqIf;
	int sockfd;

	assert(ifname != NULL);
	assert(ip != NULL);

    // Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
 		return NETHELPER_EXIT_FAILURE;
 	}

	// Copy the interface name
	strncpy(reqIf.ifr_name,ifname,sizeof(reqIf.ifr_name));
	memcpy(&(((struct sockaddr_in *)&reqIf.ifr_addr)->sin_addr),ip,sizeof(struct in_addr));
 	reqIf.ifr_addr.sa_family = AF_INET;

	if( ioctl( sockfd, SIOCSIFADDR, &reqIf ) != 0 )
	{
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}
	close(sockfd);
	return NETHELPER_EXIT_SUCCESS;
}

/*************************************************************************************************************
/ Purpose: Function that allows to set the sub-net mask of a network interface
/
/ Params: char *ifname: Interface name that we want the information from. "eth0"
/              struct in_addr *netMask: Desired new network mask
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int setNetworkMask(char *ifname, struct in_addr *netMask)
{
	struct ifreq reqIf;
	int sockfd;

	assert(ifname != NULL);
	assert(netMask != NULL);

	// Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
 		return NETHELPER_EXIT_FAILURE;
 	}

	// Copy the interface name
	strncpy(reqIf.ifr_name,ifname,sizeof(reqIf.ifr_name));
	memcpy(&(((struct sockaddr_in *)&reqIf.ifr_addr)->sin_addr),netMask,sizeof(struct in_addr));
	reqIf.ifr_addr.sa_family = AF_INET;

	if( ioctl( sockfd, SIOCSIFNETMASK, &reqIf ) != 0 )
	{
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}
	close(sockfd);
	return NETHELPER_EXIT_SUCCESS;
}


/*************************************************************************************************************
/ Purpose: Function that deletes a default gateway from the route list
/
/ Params: char *ifname: Interface name that we want to delete the default gateway from
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:  To make sure all the default gateways are removed this function has to be called
/            several times, at least until the function returns errors.
/
***********************************************************************************************************/
int removeDefaultGateway(char *ifname)
{
	struct rtentry rtEntry;
	int sockfd;

	assert(ifname != NULL);

	// Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
 		return NETHELPER_EXIT_FAILURE;
 	}
	// Clear the route struct
	memset(&rtEntry,0,sizeof(struct rtentry));
	//((struct sockaddr_in *)&reqIf.ifr_addr)->sin_family = AF_INET;
	rtEntry.rt_dst.sa_family     = AF_INET;
	rtEntry.rt_gateway.sa_family = AF_INET;
	rtEntry.rt_genmask.sa_family = AF_INET;
	rtEntry.rt_dev               = ifname;

	if( ioctl( sockfd, SIOCDELRT, &rtEntry ) != 0 )
	{
		//perror("SIOCDELRT Error");
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}
	close(sockfd);
	return NETHELPER_EXIT_SUCCESS;
}


/*************************************************************************************************************
/ Purpose: Function that adds a default gateway to the route list
/
/ Params: char *ifname: Interface name that we want to add the default gateway to
/         struct in_addr *gateway: IP address of the gateway
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int addDefaultGateway(char *ifname,struct in_addr *gateway)
{
	struct rtentry rtEntry;
	int sockfd;
	struct sockaddr_in gway;

	assert(ifname != NULL);
	assert(gateway != NULL);

	// Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
 		return NETHELPER_EXIT_FAILURE;
 	}
	// Clear the route struct
	memset(&rtEntry,0,sizeof(struct rtentry));
	rtEntry.rt_dst.sa_family = AF_INET;
	rtEntry.rt_genmask.sa_family = AF_INET;
	rtEntry.rt_dev = ifname;

	// Clear the gateway field
	memset(&gway,0,sizeof(struct sockaddr_in));
	gway.sin_family = AF_INET;
	gway.sin_addr = *gateway;
	rtEntry.rt_gateway = *((struct sockaddr *)&gway);

	rtEntry.rt_flags = RTF_UP | RTF_GATEWAY;

	int rv;
	rv = ioctl( sockfd, SIOCADDRT, &rtEntry );
	if( rv != 0 )
	{
		char errorString[100];
		sprintf( errorString, "SIOCADDRT Error sock := %d, error := %d\n", sockfd, rv );
		perror( errorString );
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}
	close(sockfd);
	return NETHELPER_EXIT_SUCCESS;

}


/*************************************************************************************************************
/ Purpose: Function that allows to read the Network Mask of a network interface
/
/ Params: char *ifname: Interface name that we want the information from. "eth0"
/              struct in_addr *netMask: Buffer to hold the Network Mask
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int getNetworkMask(char *ifname, struct in_addr *netMask)
{
	struct ifreq reqIf;
	int sockfd;

	assert(ifname != NULL);
	assert(netMask != NULL);

	memset(netMask, 0, sizeof(struct in_addr));

    // Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
   {
		perror("socket");
 		return NETHELPER_EXIT_FAILURE;
 	}

	// Copy the interface name
	strncpy(reqIf.ifr_name,ifname,sizeof(reqIf.ifr_name));
	// Get the Netwrok Mask via an ioctl
	if(ioctl(sockfd, SIOCGIFNETMASK, &reqIf) < 0)
	{
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}
   else
   {
		memcpy(netMask,&(((struct sockaddr_in *)&reqIf.ifr_addr)->sin_addr),sizeof(struct in_addr));
		close(sockfd);
		return NETHELPER_EXIT_SUCCESS;
	}
}


/*************************************************************************************************************
/ Purpose: Function that allows to read the MAC address of a network interface
/
/ Params: char *ifname: Interface name that we want the information from. "eth0"
/              unsigned char *buffer[6]: Buffer where the MAC address will be stored
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int getMACAddress(char *ifname, unsigned char buffer[6])
{
	struct ifreq reqIf;
	int sockfd;

	assert(ifname != NULL);
	assert(buffer != NULL);

	// Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
		return NETHELPER_EXIT_FAILURE;
	}

	// Copy the interface name
	strncpy(reqIf.ifr_name,ifname,sizeof(reqIf.ifr_name));
	// Get the MAC address via an ioctl
	if(ioctl(sockfd, SIOCGIFHWADDR , &reqIf) < 0)
	{
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}
	else
	{
		int i;
		unsigned char *data = (unsigned char *)&reqIf.ifr_hwaddr.sa_data; // MAC Address data
		for(i=5; i>-1; i--)
		{
			buffer[i] = *data;
			data++;
		}
		close(sockfd);
		return NETHELPER_EXIT_SUCCESS;
	}
}


/*************************************************************************************************************
/ Purpose: Function that adds a broadcast route to the route list so messages directed to 255.255.255.255
/          are sent trough the especified network interface
/
/ Params: char *ifname: Interface name that we want to add the default gateway to
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int addBroadcastRoute(char *ifname)
{
	struct rtentry rtEntry;
	int sockfd;
	struct in_addr broadcastAddr;
	struct sockaddr_in addr;
	//struct sockaddr_in gway;

	assert(ifname != NULL);

    // Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
 		return NETHELPER_EXIT_FAILURE;
 	}
	// Clear the route struct
	memset(&rtEntry,0,sizeof(struct rtentry));

	inet_aton("255.255.255.255",&broadcastAddr);

	memset(&addr,0,sizeof(struct sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_addr = broadcastAddr;
	rtEntry.rt_dst = *((struct sockaddr *)&addr);
	rtEntry.rt_genmask = *((struct sockaddr *)&addr);
	rtEntry.rt_dev = ifname;
	rtEntry.rt_flags = RTF_UP | RTF_HOST;

	if( ioctl( sockfd, SIOCADDRT, &rtEntry ) != 0 )
	{
        if (errno != EEXIST)
        {
        	char ErrStrg[100];
        	sprintf(ErrStrg, "SIOCADDRT Error: add broadcast %s", ifname );
            perror(ErrStrg);
            close(sockfd);
            return NETHELPER_EXIT_FAILURE;
        }
	}
	close(sockfd);
	return NETHELPER_EXIT_SUCCESS;

}

int addMulticastRoute(char *ifname)
{
    struct rtentry rtEntry;
    int sockfd;
    struct in_addr mcastAddr;
    struct sockaddr_in addr;
    
    assert(ifname != NULL);
    
    // Create the Socket
    if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
    {
        perror("socket");
        return NETHELPER_EXIT_FAILURE;
    }
    // Clear the route struct
    memset(&rtEntry,0,sizeof(struct rtentry));
    
    inet_aton("224.0.0.0",&mcastAddr);
    
    memset(&addr,0,sizeof(struct sockaddr_in));
    addr.sin_family = AF_INET;
    addr.sin_addr = mcastAddr;
    rtEntry.rt_dst = *((struct sockaddr *)&addr);
    rtEntry.rt_genmask = *((struct sockaddr *)&addr);
    rtEntry.rt_dev = ifname;
    rtEntry.rt_flags = RTF_UP;
    
    if( ioctl( sockfd, SIOCADDRT, &rtEntry ) != 0 )
    {
        if (errno != EEXIST)
        {
            char ErrStrg[100];
            sprintf(ErrStrg, "SIOCADDRT Error: add multicast %s", ifname );
            perror(ErrStrg);
            close(sockfd);
            return NETHELPER_EXIT_FAILURE;
        }
    }
    close(sockfd);
	return NETHELPER_EXIT_SUCCESS;
}


/*************************************************************************************************************
/ Purpose: Function that shuts down an ethernet interface
/
/ Params: char *ifname: Interface name that we want to shutdown
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int ifdown(char *ifname)
{
	struct ifreq reqIf;
	int sockfd;

	assert(ifname != NULL);

    // Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
 		return NETHELPER_EXIT_FAILURE;
 	}

	// Copy the interface name
	strncpy(reqIf.ifr_name, ifname, sizeof(reqIf.ifr_name));
	// Get the Interface flasg via an ioctl
	if(ioctl(sockfd, SIOCGIFFLAGS , &reqIf) < 0)
	{
		perror("Error in ioctl SIOCGIFFLAGS\n");
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}

	// Remove the IFF_UP flag
	reqIf.ifr_flags &= ~IFF_UP;

	if(ioctl(sockfd, SIOCSIFFLAGS , &reqIf) < 0)
	{
		perror("Error in ioctl SIOCSIFFLAGS\n");
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}
	close(sockfd);
	return NETHELPER_EXIT_SUCCESS;
}

/*************************************************************************************************************
/ Purpose: Function that brings up an ethernet interface
/
/ Params: char *ifname: Interface name that we want to bring up
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int iface_up(char *ifname)
{
	struct ifreq reqIf;
	int sockfd;

	assert(ifname != NULL);

    // Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
 		return NETHELPER_EXIT_FAILURE;
 	}

	// Copy the interface name
	strncpy(reqIf.ifr_name, ifname, sizeof(reqIf.ifr_name));
	// Get the Interface flasg via an ioctl
	if(ioctl(sockfd, SIOCGIFFLAGS , &reqIf) < 0)
	{
		perror("Error in ioctl SIOCGIFFLAGS\n");
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}

	// Remove the IFF_UP flag
	reqIf.ifr_flags |= IFF_UP;

	if(ioctl(sockfd, SIOCSIFFLAGS , &reqIf) < 0)
	{
		perror("Error in ioctl SIOCSIFFLAGS\n");
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}
	close(sockfd);
	return NETHELPER_EXIT_SUCCESS;
}


int getifconfig(char * devname, int * speed, int * duplex, int * autoneg)
{
    int sock, rc;
    struct ifreq ifr;
    struct ethtool_cmd ecmd;
    
    if ((sock = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
    {
        return -1;
    }
    
    /* Setup our control structures. */
    memset(&ifr, 0, sizeof(ifr));
    strcpy(ifr.ifr_name, devname);
    
    ecmd.cmd = ETHTOOL_GSET;
    ifr.ifr_data = (caddr_t)&ecmd;
    rc = ioctl(sock, SIOCETHTOOL, &ifr);
    close(sock);
    
    if(rc < 0)	//cannot get current device information
    {
        return -1;
    }
    
    *speed   = ecmd.speed;
    *duplex  = ecmd.duplex;
    *autoneg = ecmd.autoneg;

    return 0;
}

int getifinfo(const char *iface, IF_INFO * ifinfo)
{
    FILE *fp;
    char temp[8][64], procline[512], *proclineptr, ifaceid[33];
    int check;
    
    if ((fp=fopen(PROCNETDEV, "r"))==NULL)
    {
        printf("Error: Unable to read %s.\n", PROCNETDEV);
        return -1;
    }
    
    strncpy(ifaceid, iface, 32);
    strcat(ifaceid, ":");
    
    check = 0;
    while (fgets(procline, 512, fp)!=NULL)
    {
        sscanf(procline, "%512s", temp[0]);
        if (strncmp(ifaceid, temp[0], strlen(ifaceid))==0)
        {
            check = 1;
            break;
        }
    }
    fclose(fp);
    
    if (check==0)
    {
        printf("Requested interface \"%s\" not found.\n", iface);
        return -1;
    }
    else if (ifinfo)
    {
        strncpy(ifinfo->name, iface, 32);
        
        /* get rx and tx from procline */
        proclineptr = strchr(procline, ':');
        sscanf(proclineptr+1, "%64s %64s %64s %64s %*s %*s %*s %*s %64s %64s %64s %64s",
            temp[0], temp[1], temp[2], temp[3], temp[4], temp[5], temp[6], temp[7]);
        
        ifinfo->rx = strtoull(temp[0], (char **)NULL, 0);
        ifinfo->rxp = strtoull(temp[1], (char **)NULL, 0);
        ifinfo->rx_errs = strtoull(temp[2], (char **)NULL, 0);
        ifinfo->rx_drop = strtoull(temp[3], (char **)NULL, 0);

        ifinfo->tx = strtoull(temp[4], (char **)NULL, 0);
        ifinfo->txp = strtoull(temp[5], (char **)NULL, 0);
        ifinfo->tx_errs = strtoull(temp[6], (char **)NULL, 0);
        ifinfo->tx_drop = strtoull(temp[7], (char **)NULL, 0);
    }
    
    return 0;
}

/*  
 * Get the DNS servers from /etc/resolv.conf file on Linux  
 * */ 
int getNameServers(char * name_server, char * name_server2)
{
    char line[200];
    char cmd[100];
    char data[100];

    FILE *pFile = fopen("/etc/resolv.conf","r");
    
    int serverCount = 0;
    
    
    if (pFile == NULL)
    {
        return -1;
    }

    while (fgets(line, 200, pFile)!=NULL && serverCount <2)
    {
        sscanf(line,"%s %s",cmd,data);
        if (strcmp(cmd,"nameserver")==0)
        {
            switch (serverCount)
            {
            case 0:
                strcpy(name_server, data);
                serverCount++;
                break;
            case 1:
                strcpy(name_server2, data);
                serverCount++;
                break;
            default:
                break;
            }
        }
    }
    
    fclose (pFile);

    return serverCount;
}

int setNameServers(const char * name_server, const char * name_server2)
{
    char *lines[100];
    char newLine[200];
    char cmd[100];

    int i, lineCount = 0;
    char *currentLine;
    FILE *pFile = fopen("/etc/resolv.conf","r");
    
    int serverCount = 0;
    
    if (pFile!=NULL)
    {
        currentLine = (char *)malloc(200);
        while (fgets(currentLine,200,pFile)!=NULL && lineCount < 100)
        {
            lines[lineCount++] = currentLine;
            currentLine = (char *)malloc(200);
        }
        free(currentLine);

        fclose (pFile);
    }

    pFile = fopen("/etc/resolv.conf","w");
    for (i=0;i<lineCount;i++)
    {
        sscanf(lines[i],"%s ",cmd);
        if (strcmp(cmd,"nameserver")==0)
        {
            switch (serverCount)
            {
            case 0:
                if (name_server != 0)
                {
                    snprintf(newLine,200,"nameserver %s\n",name_server);
                    fputs(newLine,pFile);
                }
                
                serverCount++;
                break;
            case 1:
                if (name_server2 != 0)
                {
                    snprintf(newLine,200,"nameserver %s\n",name_server2);
                    fputs(newLine,pFile);
                }
                
                serverCount++;
                break;
            default:
                break;
            }
        }
        else
        {
            fputs(lines[i],pFile);
        }
    }
    fclose(pFile);
    
    // Loop to free memory
    for (i=0;i<lineCount;i++)
    {
        free(lines[i]);
    }
    
    return 0;
}


////////////////////////////////////////////////////////////////////////////////////////
// This function sends a gratuitous arp request message for the eth0 interface.
// Params: None
//
// Returns: None
//
// Comments:
//
////////////////////////////////////////////////////////////////////////////////////////
void gratuitous_arp_request( void )
{
    char *ifname = ENET_IF0_NAME;
    gratuitous_arp_request_interface(ifname);
}


////////////////////////////////////////////////////////////////////////////////////////
// This function sends a gratuitous arp request message for any interface.
// Params: ifname: interface name (i.e., eth0 or eth1).
//
// Returns: None
//
// Comments:
//
////////////////////////////////////////////////////////////////////////////////////////
void gratuitous_arp_request_interface( char * ifname )
{
    int fd;
    char arp_buff[200];
    char hw_address[ETH_ALEN];
    struct ifreq ifr;
    struct sockaddr_ll sll;
    ssize_t nbytes;
    uint32_t ip_addr;
    struct sockaddr_in sa_addr;
    
    fd = socket(PF_PACKET, SOCK_RAW, 0);
    if(fd < 0)
    {
        return;
    }
    
    strncpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
    if ((ioctl(fd, SIOCGIFINDEX, &ifr)) != 0)
    {
        close(fd);
        return;
    }
    sll.sll_family = PF_PACKET;
    sll.sll_ifindex = ifr.ifr_ifindex;
    sll.sll_halen = ETH_ALEN;
    
    /* Obtain hardware address for specified interface */
    if ((ioctl(fd, SIOCGIFHWADDR, &ifr)) != 0)
    {
        close(fd);
        return;
    }
    memcpy(hw_address, ifr.ifr_ifru.ifru_hwaddr.sa_data, ETH_ALEN);
    
    /* Obtain IP address for specified interface */
    if ((ioctl(fd, SIOCGIFADDR, &ifr)) != 0)
    {
        close(fd);
        return;
    }
    memcpy(&sa_addr, &(ifr.ifr_ifru.ifru_addr), sizeof(sa_addr));
    ip_addr = sa_addr.sin_addr.s_addr;
    
    nbytes = 0;
    memset(arp_buff+nbytes, 0xff, ETH_ALEN);
    nbytes += ETH_ALEN;
    memcpy(arp_buff+nbytes, hw_address, ETH_ALEN);
    nbytes += ETH_ALEN;
    arp_buff[nbytes++] = 0x08;
    arp_buff[nbytes++] = 0x06;
    arp_buff[nbytes++] = 0x00;
    arp_buff[nbytes++] = 0x01;
    arp_buff[nbytes++] = 0x08;
    arp_buff[nbytes++] = 0x00;
    arp_buff[nbytes++] = 0x06;
    arp_buff[nbytes++] = 0x04;
    arp_buff[nbytes++] = 0x00;
    arp_buff[nbytes++] = 0x01;
    memcpy(arp_buff+nbytes, hw_address, ETH_ALEN);
    nbytes += ETH_ALEN;
    memcpy(arp_buff+nbytes, &ip_addr, 4);
    nbytes += 4;
    memset(arp_buff+nbytes, 0xff, ETH_ALEN);
    nbytes += ETH_ALEN;
    memcpy(arp_buff+nbytes, &ip_addr, 4);
    nbytes += 4;
    
    nbytes = sendto(fd, arp_buff, nbytes, 0, (struct sockaddr *)&sll, sizeof(sll));
    
    close(fd);
}

unsigned long resolve_ip(const char *ip_addr)
{
    struct in_addr addr;
    return inet_aton(ip_addr, &addr) ? addr.s_addr : 0;
}

////////////////////////////////////////////////////////////////////////////////////////
// This function returns the IP address for a specific interface (i.e., "eth1").  This
// is an extension of the wattcp library gethostid().  The IP address returned is in
// network byte order.
//
////////////////////////////////////////////////////////////////////////////////////////
unsigned long gethostip(char *ifname)
{
    struct in_addr ip;
    return getIPAddress(ifname, &ip) ? 0 : ip.s_addr;
}

