/*
|============================================================================
|
|       Copyright (C) 2011 ProSoft Technology. All rights reserved.
|
|  File:             timer.c
|
|  Class(es):        
|
|  Inherits From:    
|
|  Summary:          
|
|  Project(s):       Common Utility
|
|  Subsystem:        ProLinx
|
|  Contributors:     Henry Yu(HYU)
|
|  Description:      This file contains the definitions needed for simple timer processing.
|
|  Notes:            
|
|
|============================================================================
|  Version     Date     Author  Change    Description
|----------------------------------------------------------------------------
|  Build     10/19/2011 HYU      Created.
|============================================================================
*/
#include <stdio.h>
#include "timer.h"
#include "plxdef.h"


enum 
{
    TS_STOPPED,
	TS_RUNNING,
	TS_TIMEOUT,
};

typedef struct
{
    long  current_value;
    long  reload_value;
    timerProc  proc;
    void * p;
}SIMPLE_TIMER_DATA;

typedef struct
{
    int  state;
    int  owned;
    SIMPLE_TIMER_DATA timer_data;
}SIMPLE_TIMER_STRUCT;


/* ---------------------------- */
/* FUNCTION PROTOTYPES          */
/* ---------------------------- */
static void UpdateTimerIxTable (void);
static SIMPLE_TIMER_DATA * simple_timer_get_struct (int timer_num);

/* ---------------------------- */
/* STATIC VARIABLES             */
/* ---------------------------- */

SIMPLE_TIMER_STRUCT S_Timers[MAX_NUM_TIMERS];
int S_UsedTimerIx[MAX_NUM_TIMERS];
int S_NumTimerIxUsed;

/* ---------------------------- */
/* MISCELLANEOUS                */
/* ---------------------------- */


/**/
/* ******************************************************************** */
/*                      GLOBAL FUNCTIONS                                */
/* ******************************************************************** */
/* ====================================================================
Function:   simple_timer_init
Parameters: init type
Returns:    N/A

This function initializes static variables used by this file.
======================================================================= */
void simple_timer_init ()
{
    int i;

    /* initialize all the timers */
    for(i=0; i<MAX_NUM_TIMERS; i++)
    {
        simple_timer_free(i);
    }

    /* keep track of how many timers are used */
    S_NumTimerIxUsed = 0;
}

/* ====================================================================
Function:   simple_timer_new
Parameters: N/A
Returns:    N/A

This function searches for the next unused timer.  If all the timers
are used, SIMPLE_TIMER_NULL is returned.
======================================================================= */
int simple_timer_new (void)
{
    int i;

    /* Make sure we have free timers */
    if(S_NumTimerIxUsed == MAX_NUM_TIMERS)
    {
        printf("ERROR: No free resource to allocate a new timer.\n");
        return(SIMPLE_TIMER_NULL);
    }

    /* search through all timers for a free one */
    for(i=0; i<MAX_NUM_TIMERS; i++)
    {
        /* we found a free timer, use it */
        if(S_Timers[i].owned == FALSE)
        {
            /* reset the timer values to 0 to start */
            simple_timer_free(i);

            /* mark the timer as used */
            S_Timers[i].owned = TRUE;
         
            /* update the table of used indexes */
            UpdateTimerIxTable();

            /* return the timer number */
            return(i);
        }
    }

    /* no timers are free */
    return(SIMPLE_TIMER_NULL);
}

/* ====================================================================
Function:   simple_timer_create
Parameters: function pointer (called when the timer times out)
Returns:    N/A

This function gets a timer from the RTA Timer Subsystem and sets the
callback function.  A fatal error occurs if a timer can't be allocated.
======================================================================= */
int simple_timer_create (long interval, timerProc proc, void * p)
{
    int timer_id;
    SIMPLE_TIMER_DATA *TimerStructPtr;

    // allocate a timer
    timer_id = simple_timer_new();

    // get the timer structure to initialize the callback function
    TimerStructPtr = simple_timer_get_struct (timer_id);
    if(TimerStructPtr == NULL)
    {
        return SIMPLE_TIMER_NULL;
    }
  
    // set the default interval for the timer
    TimerStructPtr->reload_value = interval;
    TimerStructPtr->current_value = interval;

    // set the callback function
    TimerStructPtr->proc = proc;

    // stop the timer initially
    simple_timer_stop(timer_id);

    // return the timer id
    return(timer_id);
}

/* ====================================================================
Function:   simple_timer_free
Parameters: timer number
Returns:    N/A

This function validates the timer number, then frees the timer to
unused.
======================================================================= */
void simple_timer_free (int timer_num)
{
    // make sure the timer number isn't SIMPLE_TIMER_NULL
    if(timer_num == SIMPLE_TIMER_NULL)
        return;
    
    /* validate timer_num is in range */
    if(timer_num >= MAX_NUM_TIMERS)
        return;

    /* free the timer */
    S_Timers[timer_num].owned = FALSE;
    S_Timers[timer_num].state = TS_STOPPED;
    S_Timers[timer_num].timer_data.current_value = 0;
    S_Timers[timer_num].timer_data.reload_value  = 0;
    S_Timers[timer_num].timer_data.proc = NULL;
}

/* ====================================================================
Function:   simple_timer_reset
Parameters: timer number
Returns:    N/A

This function validates the timer number, then sets the current value
to the reload value.
======================================================================= */
void simple_timer_reset (int timer_num)
{
    // make sure the timer number isn't SIMPLE_TIMER_NULL
    if(timer_num == SIMPLE_TIMER_NULL)
        return;
    
    /* validate timer_num is in range */
    if(timer_num >= MAX_NUM_TIMERS)
        return;

    /* only reset the timer if it is owned */
    if(S_Timers[timer_num].owned == TRUE)
    {
        /* set the current value to the reload value */
        S_Timers[timer_num].timer_data.current_value = S_Timers[timer_num].timer_data.reload_value;

        /* set the state back to running */
        S_Timers[timer_num].state = TS_RUNNING;
    }
}

/* ====================================================================
Function:    simple_timer_restart
Parameters:  timer number
             new timeout value
Returns:     N/A

This function sets the timeout value to the passed value and starts the timer.
======================================================================= */
void simple_timer_restart (int timer_num, long newInterval)
{
   /* validate timer_num is in range */
   if(timer_num >= MAX_NUM_TIMERS)
      return;

   /* only start the timer if it is owned */
   if(S_Timers[timer_num].owned == TRUE)
   {
      /* set the current value to the reload value */
      S_Timers[timer_num].timer_data.current_value = newInterval;
      S_Timers[timer_num].timer_data.reload_value  = newInterval;

      /* set the state back to running */
      S_Timers[timer_num].state = TS_RUNNING;
   }
}

/* ====================================================================
Function:   simple_timer_start
Parameters: timer number
Returns:    N/A

This function validates the timer number, then starts the timer.
======================================================================= */
void simple_timer_start (int timer_num)
{
    // make sure the timer number isn't SIMPLE_TIMER_NULL
    if(timer_num == SIMPLE_TIMER_NULL)
        return;
    
    /* validate timer_num is in range */
    if(timer_num >= MAX_NUM_TIMERS)
        return;

    /* only start the timer if it is owned */
    if(S_Timers[timer_num].owned == TRUE)
    {
        /* set the state back to running */
        S_Timers[timer_num].state = TS_RUNNING;
    }
}

/* ====================================================================
Function:   simple_timer_stop
Parameters: timer number
Returns:    N/A

This function validates the timer number, then stops the timer.
======================================================================= */
void simple_timer_stop (int timer_num)
{
    // make sure the timer number isn't SIMPLE_TIMER_NULL
    if(timer_num == SIMPLE_TIMER_NULL)
        return;
    
    /* validate timer_num is in range */
    if(timer_num >= MAX_NUM_TIMERS)
        return;

    /* only start the timer if it is owned */
    if(S_Timers[timer_num].owned == TRUE)
    {
        /* set the state to stopped */
        S_Timers[timer_num].state = TS_STOPPED;
    }
}

/* ====================================================================
Function:   simple_timer_running
Parameters: timer number
Returns:    TRUE  - timer is running
            FALSE - timer isn't running

This function validates the timer number and returns TRUE if the timer
is running.
======================================================================= */
int simple_timer_running (int timer_num)
{
    // make sure the timer number isn't SIMPLE_TIMER_NULL
    if(timer_num == SIMPLE_TIMER_NULL)
        return(FALSE);
    
    /* validate timer_num is in range */
    if(timer_num >= MAX_NUM_TIMERS)
        return(FALSE);

    /* only start the timer if it is owned */
    if( (S_Timers[timer_num].owned == TRUE) &&
        (S_Timers[timer_num].state != TS_STOPPED))
    {
        // we are running
        return(TRUE);
    }

    return(FALSE);
}


/* ====================================================================
Function:   simple_timer_get_struct
Parameters: timer number
Returns:    pointer to the timer data element, NULL on error

This function validates the timer number is valid and owned, then
returns the timer structure.
======================================================================= */
SIMPLE_TIMER_DATA * simple_timer_get_struct (int timer_num)
{
    // make sure the timer number isn't SIMPLE_TIMER_NULL
    if(timer_num == SIMPLE_TIMER_NULL)
        return(NULL);
   
    /* validate timer_num is in range */
    if(timer_num >= MAX_NUM_TIMERS)
        return(NULL);

    /* only start the timer if it is owned */
    if(S_Timers[timer_num].owned == TRUE)
    {
        /* set the state back to running */
        return(&S_Timers[timer_num].timer_data);
    }

    /* if the timer isn't owned, return NULL */
    return(NULL);
}

/* ====================================================================
Function:   simple_timer_process
Parameters: N/A
Returns:    N/A

This function processes all timers.  Flags are incremented on timeout.
Function pointers are called on timeouts.
======================================================================= */
void simple_timer_process (long time_passed)
{
    int i, timer_ix;
    long current_tv;

    /* if ticks_passed is 0, return */
    if(time_passed == 0)
        return;

    /* only process timers in our lookup table */
    for(i=0; i<S_NumTimerIxUsed; i++)
    {
        /* get the next timer index to access */
        timer_ix = S_UsedTimerIx[i];

        /* only check the timer if it is running */
        if(S_Timers[timer_ix].state == TS_RUNNING)
        {
            current_tv = S_Timers[timer_ix].timer_data.current_value - time_passed;
            /* see if ticks_passed causes us to transition to timed out */
            if(current_tv <= 0)
            {
                /* transition to timed out */
                S_Timers[timer_ix].state = TS_TIMEOUT;

                /* set the current value to 0 */
                S_Timers[timer_ix].timer_data.current_value = 0;

                /* if available call the timeout function pointer */
                if(S_Timers[timer_ix].timer_data.proc != NULL)
                {
                    (*S_Timers[timer_ix].timer_data.proc)(timer_ix, S_Timers[timer_ix].timer_data.p);
                }
            }
    
            /* we didn't timeout, decrement our current count */
            else
            {
                S_Timers[S_UsedTimerIx[i]].timer_data.current_value = current_tv;
            }
        }
    }
}

/**/
/* ******************************************************************** */
/*                      LOCAL FUNCTIONS                                 */
/* ******************************************************************** */
/* ====================================================================
Function:   UpdateTimerIxTable
Parameters: N/A
Returns:    N/A

A table of used indexes is implemented to shorten the process time for
timers.  Only used indexes are updated in the timer process.
======================================================================= */
void UpdateTimerIxTable (void)
{
    int i;

    /* if a timer is used, add it to the table of used indexes */
    for(i=0, S_NumTimerIxUsed = 0; i<MAX_NUM_TIMERS; i++)
    {
        /* if a timer is owned, add the index to the table */
        if(S_Timers[i].owned == TRUE)
        {
            /* add the index to the lookup table */
            S_UsedTimerIx[S_NumTimerIxUsed] = i;

            /* increment the number of used indexes */
            S_NumTimerIxUsed++;
        }
    }
}


/* *********** */
/* END OF FILE */
/* *********** */
